<?php
if (!defined('ABSPATH')) { exit; }

class PI_Parser {
    /**
     * Parse Lighthouse JSON and return structured results with plugin mapping.
     * @param string $json
     * @return array{errors:array, sections:array}
     */
    public static function parse($json) {
        $errors = [];
        $warnings = [];
        $sections = [];

        // Handle null input
        if ($json === null) {
            $errors[] = __('No input provided. Paste the Lighthouse JSON report.', 'performance-insights');
            return compact('errors', 'warnings', 'sections');
        }

        // Trim whitespace and check for empty input
        $json = trim($json);
        if (empty($json)) {
            $errors[] = __('No input provided. Paste the Lighthouse JSON report.', 'performance-insights');
            return compact('errors', 'warnings', 'sections');
        }

        $data = json_decode($json, true);
        if (json_last_error() !== JSON_ERROR_NONE || !is_array($data)) {
            $json_error = json_last_error_msg();
            $errors[] = sprintf(__('Invalid JSON: %s. Please paste a valid Lighthouse JSON report.', 'performance-insights'), $json_error);
            return compact('errors', 'warnings', 'sections');
        }

        // Check for runtime errors in the Lighthouse report but continue processing if there's audit data
        if (isset($data['runtimeError'])) {
            $runtime_error = $data['runtimeError'];
            $error_code = $runtime_error['code'] ?? 'UNKNOWN';
            $error_message = $runtime_error['message'] ?? 'Unknown error occurred';
            
            // Add warning but don't stop processing if we have audits
            if (!isset($data['audits']) || empty($data['audits'])) {
                $errors[] = sprintf(
                    __('Lighthouse report contains a runtime error (%s): %s and no audit data is available.', 'performance-insights'),
                    $error_code,
                    $error_message
                );
                return compact('errors', 'warnings', 'sections');
            } else {
                // Add warning but continue processing
                $warnings[] = sprintf(
                    __('Warning: Lighthouse report contains a runtime error (%s): %s. Some data may be incomplete.', 'performance-insights'),
                    $error_code,
                    $error_message
                );
            }
        }

        if (!isset($data['audits']) || !is_array($data['audits'])) {
            $errors[] = __('JSON does not contain expected "audits" field. This may be an incomplete or corrupted Lighthouse report.', 'performance-insights');
            return compact('errors', 'warnings', 'sections');
        }

        $audits = $data['audits'];

        // Check if audits is empty (which can happen with runtime errors)
        if (empty($audits)) {
            $errors[] = __('Lighthouse report contains no audit data. This usually indicates the audit failed to complete.', 'performance-insights');
            return compact('errors', 'warnings', 'sections');
        }

        // Extract all problematic scripts with enhanced data
        $all_scripts = [];
        
        // Try to extract from each audit type, but don't fail if some are missing
        try {
            $all_scripts = array_merge($all_scripts, self::extract_render_blocking($audits));
        } catch (Exception $e) {
            // Log but continue
        }
        
        try {
            $all_scripts = array_merge($all_scripts, self::extract_unused_js($audits));
        } catch (Exception $e) {
            // Log but continue
        }
        
        try {
            $all_scripts = array_merge($all_scripts, self::extract_total_byte_weight($audits));
        } catch (Exception $e) {
            // Log but continue
        }
        
        try {
            $all_scripts = array_merge($all_scripts, self::extract_bootup_time($audits));
        } catch (Exception $e) {
            // Log but continue
        }

        // Map scripts to their sources and add insights
        $enhanced_scripts = self::map_script_sources($all_scripts);
        
        // If no problematic scripts found, provide basic report info
        if (empty($enhanced_scripts)) {
            $sections['info'] = [
                'title' => __('Lighthouse Report Information', 'performance-insights'),
                'message' => __('No significant performance issues detected in the provided Lighthouse report.', 'performance-insights'),
                'details' => [
                    'url' => $data['requestedUrl'] ?? $data['finalUrl'] ?? 'Unknown',
                    'lighthouse_version' => $data['lighthouseVersion'] ?? 'Unknown',
                    'fetch_time' => $data['fetchTime'] ?? 'Unknown',
                    'user_agent' => $data['userAgent'] ?? 'Unknown',
                    'audit_count' => count($audits)
                ]
            ];
        } else {
            $sections['scripts'] = [
                'title' => __('Performance Issues by Plugin/Source', 'performance-insights'),
                'scripts' => $enhanced_scripts,
                'summary' => sprintf(__('Found %d problematic scripts from %d different sources', 'performance-insights'), 
                    count($enhanced_scripts), 
                    count(array_unique(array_column($enhanced_scripts, 'source_name')))
                )
            ];
        }

        return compact('errors', 'warnings', 'sections');
    }

    private static function extract_render_blocking($audits) {
        $key = 'render-blocking-resources';
        $scripts = [];
        
        if (!isset($audits[$key])) return $scripts;
        $a = $audits[$key];
        $items = $a['details']['items'] ?? [];

        foreach ($items as $item) {
            $url = $item['url'] ?? '';
            if (self::is_script_resource($url)) {
                $scripts[] = [
                    'url' => $url,
                    'issue_type' => 'render-blocking',
                    'impact_ms' => $item['wastedMs'] ?? 0,
                    'impact_bytes' => $item['totalBytes'] ?? $item['transferSize'] ?? 0,
                    'raw_data' => $item
                ];
            }
        }
        return $scripts;
    }

    private static function extract_unused_js($audits) {
        $key = 'unused-javascript';
        $scripts = [];
        
        if (!isset($audits[$key])) return $scripts;
        $a = $audits[$key];
        $items = $a['details']['items'] ?? [];

        foreach ($items as $item) {
            $url = $item['url'] ?? '';
            $scripts[] = [
                'url' => $url,
                'issue_type' => 'unused-js',
                'impact_ms' => 0,
                'impact_bytes' => $item['wastedBytes'] ?? 0,
                'waste_percent' => isset($item['wastedPercent']) ? round($item['wastedPercent'] * 100, 1) : 0,
                'raw_data' => $item
            ];
        }
        return $scripts;
    }

    private static function extract_total_byte_weight($audits) {
        $key = 'total-byte-weight';
        $scripts = [];
        
        if (!isset($audits[$key])) return $scripts;
        $a = $audits[$key];
        $items = $a['details']['items'] ?? [];

        foreach ($items as $item) {
            $url = $item['url'] ?? '';
            if (self::is_script_resource($url)) {
                $scripts[] = [
                    'url' => $url,
                    'issue_type' => 'large-payload',
                    'impact_ms' => 0,
                    'impact_bytes' => $item['totalBytes'] ?? $item['transferSize'] ?? 0,
                    'raw_data' => $item
                ];
            }
        }
        return $scripts;
    }

    private static function extract_bootup_time($audits) {
        $key = 'bootup-time';
        $scripts = [];
        
        if (!isset($audits[$key])) return $scripts;
        $a = $audits[$key];
        $items = $a['details']['items'] ?? [];

        foreach ($items as $item) {
            $url = $item['url'] ?? '';
            $scripts[] = [
                'url' => $url,
                'issue_type' => 'bootup-heavy',
                'impact_ms' => $item['total'] ?? $item['duration'] ?? 0,
                'impact_bytes' => 0,
                'raw_data' => $item
            ];
        }
        return $scripts;
    }

    private static function map_script_sources($scripts) {
        $enhanced_scripts = [];
        $plugins = get_plugins();
        $site_url = site_url();
        $wp_content_url = content_url();
        
        foreach ($scripts as $script) {
            $url = $script['url'];
            $source_info = self::identify_source($url, $plugins, $site_url, $wp_content_url);
            
            $enhanced_script = array_merge($script, $source_info);
            $enhanced_script['severity'] = self::calculate_severity($script);
            
            // Avoid duplicates by URL + issue type
            $key = md5($url . $script['issue_type']);
            if (!isset($enhanced_scripts[$key])) {
                $enhanced_scripts[$key] = $enhanced_script;
            }
        }
        
        return array_values($enhanced_scripts);
    }

    private static function identify_source($url, $plugins, $site_url, $wp_content_url) {
        // Check if it's a local resource
        if (strpos($url, $site_url) === false && !self::is_relative_url($url)) {
            return [
                'source_type' => 'external',
                'source_name' => self::extract_domain($url),
                'source_slug' => null,
                'plugin_info' => null
            ];
        }

        // Normalize URL for local analysis
        $path = str_replace([$site_url, $wp_content_url], '', $url);
        $path = ltrim($path, '/');

        // Check for plugin source
        if (strpos($path, 'plugins/') !== false) {
            preg_match('/plugins\/([^\/]+)/', $path, $matches);
            if (!empty($matches[1])) {
                $plugin_slug = $matches[1];
                $plugin_info = self::find_plugin_info($plugin_slug, $plugins);
                
                return [
                    'source_type' => 'plugin',
                    'source_name' => $plugin_info['name'] ?? $plugin_slug,
                    'source_slug' => $plugin_slug,
                    'plugin_info' => $plugin_info
                ];
            }
        }

        // Check for theme source
        if (strpos($path, 'themes/') !== false) {
            preg_match('/themes\/([^\/]+)/', $path, $matches);
            if (!empty($matches[1])) {
                $theme_slug = $matches[1];
                return [
                    'source_type' => 'theme',
                    'source_name' => $theme_slug === get_stylesheet() ? __('Active Theme', 'performance-insights') : $theme_slug,
                    'source_slug' => $theme_slug,
                    'plugin_info' => null
                ];
            }
        }

        // WordPress core or unknown
        if (strpos($path, 'wp-includes/') !== false || strpos($path, 'wp-admin/') !== false) {
            return [
                'source_type' => 'core',
                'source_name' => __('WordPress Core', 'performance-insights'),
                'source_slug' => null,
                'plugin_info' => null
            ];
        }

        return [
            'source_type' => 'unknown',
            'source_name' => __('Unknown Source', 'performance-insights'),
            'source_slug' => null,
            'plugin_info' => null
        ];
    }

    private static function find_plugin_info($slug, $plugins) {
        foreach ($plugins as $plugin_file => $plugin_data) {
            if (strpos($plugin_file, $slug . '/') === 0) {
                return [
                    'name' => $plugin_data['Name'],
                    'version' => $plugin_data['Version'],
                    'file' => $plugin_file,
                    'active' => is_plugin_active($plugin_file)
                ];
            }
        }
        return null;
    }

    private static function calculate_severity($script) {
        $bytes = $script['impact_bytes'] ?? 0;
        $ms = $script['impact_ms'] ?? 0;
        
        // Critical thresholds
        if ($bytes >= 200 * 1024 || $ms >= 500) return 'critical';
        
        // Moderate thresholds
        if ($bytes >= 50 * 1024 || $ms >= 150) return 'moderate';
        
        return 'fine';
    }

    private static function is_script_resource($url) {
        $js_extensions = ['.js', '.mjs', '.ts'];
        foreach ($js_extensions as $ext) {
            if (strpos($url, $ext) !== false) return true;
        }
        return false;
    }

    private static function is_relative_url($url) {
        return !preg_match('/^https?:\/\//', $url);
    }

    private static function extract_domain($url) {
        $parsed = parse_url($url);
        return $parsed['host'] ?? __('External', 'performance-insights');
    }

    // Utility formatters
    public static function format_bytes($bytes) {
        if (!is_numeric($bytes)) return '';
        $units = ['B','KB','MB','GB'];
        $i = 0;
        while ($bytes >= 1024 && $i < count($units)-1) { $bytes /= 1024; $i++; }
        return sprintf('%.1f %s', $bytes, $units[$i]);
    }

    public static function format_ms($ms) {
        if (!is_numeric($ms)) return '';
        return sprintf('%d ms', round($ms));
    }
}
