<?php
if (!defined('ABSPATH')) { exit; }

class PI_Insights {
    
    /**
     * Generate exclusion suggestions for a script based on its source and type
     * @param array $script Enhanced script data with source mapping
     * @return string Exclusion suggestion
     */
    public static function get_exclusion_suggestion($script) {
        $source_type = $script['source_type'] ?? '';
        $source_name = $script['source_name'] ?? '';
        $source_slug = $script['source_slug'] ?? '';
        $issue_type = $script['issue_type'] ?? '';
        
        // External scripts - general advice
        if ($source_type === 'external') {
            return self::get_external_suggestion($source_name, $issue_type);
        }
        
        // WordPress core - usually can't exclude
        if ($source_type === 'core') {
            return __('WordPress core script - review if truly necessary', 'performance-insights');
        }
        
        // Theme scripts
        if ($source_type === 'theme') {
            return self::get_theme_suggestion($issue_type);
        }
        
        // Plugin scripts - the main focus
        if ($source_type === 'plugin') {
            return self::get_plugin_suggestion($source_slug, $source_name, $issue_type);
        }
        
        return __('Manual review needed - unknown source', 'performance-insights');
    }
    
    private static function get_external_suggestion($domain, $issue_type) {
        $suggestions = [
            'google-analytics.com' => __('Load Google Analytics async or use gtag', 'performance-insights'),
            'googletagmanager.com' => __('Load GTM async or defer', 'performance-insights'),
            'facebook.net' => __('Load Facebook Pixel async or exclude on non-conversion pages', 'performance-insights'),
            'doubleclick.net' => __('Consider excluding ads on admin/checkout pages', 'performance-insights'),
            'googlesyndication.com' => __('Load AdSense async or exclude on specific page types', 'performance-insights'),
            'youtube.com' => __('Load YouTube embeds on user interaction', 'performance-insights'),
            'vimeo.com' => __('Load Vimeo embeds on user interaction', 'performance-insights'),
        ];
        
        foreach ($suggestions as $service => $suggestion) {
            if (strpos($domain, $service) !== false) {
                return $suggestion;
            }
        }
        
        if ($issue_type === 'render-blocking') {
            return sprintf(__('Load %s async/defer or exclude on non-essential pages', 'performance-insights'), $domain);
        }
        
        return sprintf(__('Review %s - consider async loading or conditional exclusion', 'performance-insights'), $domain);
    }
    
    private static function get_theme_suggestion($issue_type) {
        switch ($issue_type) {
            case 'unused-js':
                return __('Exclude theme JS on pages where features aren\'t used (archives, simple posts)', 'performance-insights');
            case 'render-blocking':
                return __('Move theme scripts to footer or load conditionally', 'performance-insights');
            case 'large-payload':
                return __('Consider minifying theme assets or splitting into smaller files', 'performance-insights');
            default:
                return __('Review theme script necessity on different page types', 'performance-insights');
        }
    }
    
    private static function get_plugin_suggestion($slug, $name, $issue_type) {
        // Known plugin patterns and their typical exclusion contexts
        $plugin_rules = [
            // Form plugins
            'contact-form-7' => [
                'name' => 'Contact Form 7',
                'exclude_on' => __('Exclude on: category/tag archives, author pages, search results', 'performance-insights'),
                'keep_on' => __('Keep on: pages with contact forms', 'performance-insights')
            ],
            'wpforms-lite' => [
                'name' => 'WPForms',
                'exclude_on' => __('Exclude on: blog archives, posts without forms', 'performance-insights'),
                'keep_on' => __('Keep on: contact pages, landing pages with forms', 'performance-insights')
            ],
            'ninja-forms' => [
                'name' => 'Ninja Forms',
                'exclude_on' => __('Exclude on: category archives, tag pages, author pages', 'performance-insights'),
                'keep_on' => __('Keep on: pages containing forms', 'performance-insights')
            ],
            
            // Page builders
            'elementor' => [
                'name' => 'Elementor',
                'exclude_on' => __('Exclude on: category/tag archives, search results, posts not built with Elementor', 'performance-insights'),
                'keep_on' => __('Keep on: pages built with Elementor page builder', 'performance-insights')
            ],
            'divi-builder' => [
                'name' => 'Divi Builder',
                'exclude_on' => __('Exclude on: blog archives, posts without Divi layouts', 'performance-insights'),
                'keep_on' => __('Keep on: pages using Divi Builder', 'performance-insights')
            ],
            'beaver-builder' => [
                'name' => 'Beaver Builder',
                'exclude_on' => __('Exclude on: archives, posts not using Beaver Builder', 'performance-insights'),
                'keep_on' => __('Keep on: pages built with Beaver Builder', 'performance-insights')
            ],
            
            // E-commerce
            'woocommerce' => [
                'name' => 'WooCommerce',
                'exclude_on' => __('Exclude on: blog posts, non-shop pages, about/contact pages', 'performance-insights'),
                'keep_on' => __('Keep on: shop, product, cart, checkout, account pages', 'performance-insights')
            ],
            'easy-digital-downloads' => [
                'name' => 'Easy Digital Downloads',
                'exclude_on' => __('Exclude on: blog content, non-store pages', 'performance-insights'),
                'keep_on' => __('Keep on: download pages, checkout, account areas', 'performance-insights')
            ],
            
            // SEO plugins
            'wordpress-seo' => [
                'name' => 'Yoast SEO',
                'exclude_on' => __('Exclude frontend scripts on: admin pages (keep backend functionality)', 'performance-insights'),
                'keep_on' => __('Keep on: most pages (essential for SEO)', 'performance-insights')
            ],
            'all-in-one-seo-pack' => [
                'name' => 'All in One SEO',
                'exclude_on' => __('Review frontend scripts - may be excludable on most pages', 'performance-insights'),
                'keep_on' => __('Keep backend functionality', 'performance-insights')
            ],
            
            // Social/sharing
            'social-warfare' => [
                'name' => 'Social Warfare',
                'exclude_on' => __('Exclude on: pages without social sharing (archives, admin pages)', 'performance-insights'),
                'keep_on' => __('Keep on: blog posts, pages where sharing is needed', 'performance-insights')
            ],
            
            // Sliders/galleries
            'revslider' => [
                'name' => 'Revolution Slider',
                'exclude_on' => __('Exclude on: pages without sliders (most pages)', 'performance-insights'),
                'keep_on' => __('Keep on: homepage, landing pages with sliders', 'performance-insights')
            ],
            'smart-slider-3' => [
                'name' => 'Smart Slider 3',
                'exclude_on' => __('Exclude on: pages without sliders', 'performance-insights'),
                'keep_on' => __('Keep on: specific pages using sliders', 'performance-insights')
            ],
            
            // Analytics/tracking
            'google-analytics-for-wordpress' => [
                'name' => 'MonsterInsights',
                'exclude_on' => __('Consider excluding on: admin pages, thank you pages', 'performance-insights'),
                'keep_on' => __('Keep on: most frontend pages for tracking', 'performance-insights')
            ],
            
            // Chat/support
            'wp-live-chat-support' => [
                'name' => 'WP Live Chat',
                'exclude_on' => __('Exclude on: checkout pages, admin areas', 'performance-insights'),
                'keep_on' => __('Keep on: support pages, product pages', 'performance-insights')
            ],
            
            // Performance/caching (ironic)
            'wp-rocket' => [
                'name' => 'WP Rocket',
                'exclude_on' => __('Review admin scripts - frontend should be minimal', 'performance-insights'),
                'keep_on' => __('Essential for caching functionality', 'performance-insights')
            ]
        ];
        
        // Check for exact slug match
        if (isset($plugin_rules[$slug])) {
            $rule = $plugin_rules[$slug];
            return $rule['exclude_on'] . ' | ' . $rule['keep_on'];
        }
        
        // Check for partial matches in slug or name
        foreach ($plugin_rules as $pattern => $rule) {
            if (strpos($slug, $pattern) !== false || strpos(strtolower($name), $pattern) !== false) {
                return $rule['exclude_on'] . ' | ' . $rule['keep_on'];
            }
        }
        
        // Generic suggestions based on issue type
        return self::get_generic_plugin_suggestion($name, $issue_type);
    }
    
    private static function get_generic_plugin_suggestion($name, $issue_type) {
        switch ($issue_type) {
            case 'unused-js':
                return sprintf(__('%s: Exclude on pages where plugin features aren\'t used', 'performance-insights'), $name);
            case 'render-blocking':
                return sprintf(__('%s: Move to footer or load conditionally based on page content', 'performance-insights'), $name);
            case 'large-payload':
                return sprintf(__('%s: Consider excluding on mobile or non-essential pages', 'performance-insights'), $name);
            case 'bootup-heavy':
                return sprintf(__('%s: Defer loading or exclude on pages with fast interaction needs', 'performance-insights'), $name);
            default:
                return sprintf(__('%s: Review necessity on different page types (archives, posts, pages)', 'performance-insights'), $name);
        }
    }
    
    /**
     * Get priority score for sorting (higher = more important to address)
     * @param array $script
     * @return int Priority score
     */
    public static function get_priority_score($script) {
        $score = 0;
        
        // Issue type weights
        $issue_weights = [
            'render-blocking' => 100,
            'unused-js' => 80,
            'bootup-heavy' => 70,
            'large-payload' => 60
        ];
        
        $score += $issue_weights[$script['issue_type']] ?? 0;
        
        // Severity weights
        $severity_weights = [
            'critical' => 50,
            'moderate' => 30,
            'fine' => 10
        ];
        
        $score += $severity_weights[$script['severity']] ?? 0;
        
        // Impact weights
        $bytes = $script['impact_bytes'] ?? 0;
        $ms = $script['impact_ms'] ?? 0;
        
        $score += min($bytes / 1024, 100); // Up to 100 points for size
        $score += min($ms / 10, 50); // Up to 50 points for time
        
        return $score;
    }
    
    /**
     * Generate summary insights for the entire analysis
     * @param array $scripts
     * @return array Summary data
     */
    public static function generate_summary($scripts) {
        if (empty($scripts)) {
            return [
                'total_scripts' => 0,
                'top_issues' => [],
                'plugin_breakdown' => [],
                'recommendations' => []
            ];
        }
        
        // Sort by priority
        usort($scripts, function($a, $b) {
            return self::get_priority_score($b) - self::get_priority_score($a);
        });
        
        // Get top 3 issues
        $top_issues = array_slice($scripts, 0, 3);
        
        // Plugin breakdown
        $plugin_breakdown = [];
        foreach ($scripts as $script) {
            $source = $script['source_name'];
            if (!isset($plugin_breakdown[$source])) {
                $plugin_breakdown[$source] = [
                    'count' => 0,
                    'total_bytes' => 0,
                    'total_ms' => 0,
                    'severity_counts' => ['critical' => 0, 'moderate' => 0, 'fine' => 0]
                ];
            }
            
            $plugin_breakdown[$source]['count']++;
            $plugin_breakdown[$source]['total_bytes'] += $script['impact_bytes'] ?? 0;
            $plugin_breakdown[$source]['total_ms'] += $script['impact_ms'] ?? 0;
            $plugin_breakdown[$source]['severity_counts'][$script['severity']]++;
        }
        
        // Sort plugin breakdown by impact
        uasort($plugin_breakdown, function($a, $b) {
            return ($b['total_bytes'] + $b['total_ms'] * 1000) - ($a['total_bytes'] + $a['total_ms'] * 1000);
        });
        
        return [
            'total_scripts' => count($scripts),
            'top_issues' => $top_issues,
            'plugin_breakdown' => array_slice($plugin_breakdown, 0, 5, true), // Top 5 plugins
            'recommendations' => self::generate_top_recommendations($scripts)
        ];
    }
    
    private static function generate_top_recommendations($scripts) {
        $recommendations = [];
        
        // Find the most impactful plugins to exclude
        $plugin_impacts = [];
        foreach ($scripts as $script) {
            $source = $script['source_name'];
            if ($script['source_type'] === 'plugin') {
                if (!isset($plugin_impacts[$source])) {
                    $plugin_impacts[$source] = 0;
                }
                $plugin_impacts[$source] += self::get_priority_score($script);
            }
        }
        
        arsort($plugin_impacts);
        $top_plugins = array_slice(array_keys($plugin_impacts), 0, 3, true);
        
        foreach ($top_plugins as $plugin) {
            $recommendations[] = sprintf(
                __('Consider excluding %s on pages where it\'s not needed for maximum performance gain', 'performance-insights'),
                $plugin
            );
        }
        
        return $recommendations;
    }
}
