/* Performance Insights - Plugin Exclusion Analyzer - Admin JavaScript */

jQuery(document).ready(function($) {
    
    // Form submission loading state
    $('.pi-form').on('submit', function() {
        var $form = $(this);
        var $submit = $form.find('input[type="submit"]');
        
        // Validate textarea is not empty
        var jsonContent = $('#lighthouse_json').val().trim();
        if (!jsonContent) {
            alert('Please paste your Lighthouse JSON report before analyzing.');
            return false;
        }
        
        // Basic JSON validation
        try {
            JSON.parse(jsonContent);
        } catch (e) {
            if (!confirm('The content doesn\'t appear to be valid JSON. Do you want to continue anyway?')) {
                return false;
            }
        }
        
        // Add loading state
        $submit.prop('disabled', true).val('Analyzing...');
        $form.addClass('pi-loading');
        
        return true;
    });
    
    // Table sorting functionality
    var currentSort = { column: null, direction: 'asc' };
    
    $('.pi-sortable').on('click', function() {
        var $th = $(this);
        var column = $th.data('sort');
        var $table = $th.closest('table');
        var $tbody = $table.find('tbody');
        var rows = $tbody.find('tr').toArray();
        
        // Determine sort direction
        if (currentSort.column === column) {
            currentSort.direction = currentSort.direction === 'asc' ? 'desc' : 'asc';
        } else {
            currentSort.direction = 'asc';
        }
        currentSort.column = column;
        
        // Update visual indicators
        $('.pi-sortable').removeClass('sorted-asc sorted-desc');
        $th.addClass('sorted-' + currentSort.direction);
        
        // Sort rows
        rows.sort(function(a, b) {
            var aVal = getSortValue($(a), column);
            var bVal = getSortValue($(b), column);
            
            if (currentSort.direction === 'asc') {
                return aVal > bVal ? 1 : (aVal < bVal ? -1 : 0);
            } else {
                return aVal < bVal ? 1 : (aVal > bVal ? -1 : 0);
            }
        });
        
        // Reorder DOM
        $tbody.empty().append(rows);
    });
    
    function getSortValue($row, column) {
        switch (column) {
            case 'url':
                return $row.find('.pi-script-url code').text().toLowerCase();
            case 'source':
                return $row.find('.pi-source-type').text().toLowerCase();
            case 'issue':
                return $row.find('.pi-issue-badge').text().toLowerCase();
            case 'impact':
                // Sort by numeric impact (bytes + ms converted to comparable values)
                var bytesText = $row.find('.pi-impact-bytes').text();
                var msText = $row.find('.pi-impact-ms').text();
                var bytes = parseFloat(bytesText.replace(/[^\d.]/g, '')) || 0;
                var ms = parseFloat(msText.replace(/[^\d.]/g, '')) || 0;
                
                // Convert KB/MB to bytes for comparison
                if (bytesText.includes('KB')) bytes *= 1024;
                if (bytesText.includes('MB')) bytes *= 1024 * 1024;
                
                return bytes + (ms * 1000); // Weight ms higher for sorting
            default:
                return '';
        }
    }
    
    // Filtering functionality
    $('#pi-filter-source, #pi-filter-severity').on('change', function() {
        var sourceFilter = $('#pi-filter-source').val();
        var severityFilter = $('#pi-filter-severity').val();
        
        $('.pi-scripts-table tbody tr').each(function() {
            var $row = $(this);
            var sourceType = $row.data('source-type');
            var severity = $row.data('severity');
            var show = true;
            
            if (sourceFilter && sourceType !== sourceFilter) {
                show = false;
            }
            
            if (severityFilter && severity !== severityFilter) {
                show = false;
            }
            
            $row.toggleClass('pi-filtered', !show);
        });
        
        updateVisibleCount();
    });
    
    function updateVisibleCount() {
        var total = $('.pi-scripts-table tbody tr').length;
        var visible = $('.pi-scripts-table tbody tr:not(.pi-filtered)').length;
        
        if (visible !== total) {
            if (!$('.pi-filter-info').length) {
                $('.pi-table-controls').append('<span class="pi-filter-info"></span>');
            }
            $('.pi-filter-info').text('Showing ' + visible + ' of ' + total + ' scripts');
        } else {
            $('.pi-filter-info').remove();
        }
    }
    
    // Copy script URL on click
    $(document).on('click', '.pi-script-url code', function() {
        var $code = $(this);
        var text = $code.attr('title') || $code.text(); // Use full URL from title
        
        // Try to copy to clipboard
        if (navigator.clipboard && navigator.clipboard.writeText) {
            navigator.clipboard.writeText(text).then(function() {
                showCopyFeedback($code, 'Copied!');
            });
        } else {
            // Fallback for older browsers
            var $temp = $('<textarea>');
            $('body').append($temp);
            $temp.val(text).select();
            document.execCommand('copy');
            $temp.remove();
            showCopyFeedback($code, 'Copied!');
        }
    });
    
    function showCopyFeedback($element, message) {
        var originalText = $element.text();
        $element.text(message).addClass('pi-copied');
        
        setTimeout(function() {
            $element.text(originalText).removeClass('pi-copied');
        }, 1500);
    }
    
    // CSV Export functionality
    $('#pi-export-csv').on('click', function() {
        exportTableToCSV();
    });
    
    function exportTableToCSV() {
        var csv = [];
        var headers = [];
        
        // Get headers
        $('.pi-scripts-table thead th').each(function() {
            headers.push('"' + $(this).text().replace(/"/g, '""') + '"');
        });
        csv.push(headers.join(','));
        
        // Get visible rows
        $('.pi-scripts-table tbody tr:not(.pi-filtered)').each(function() {
            var row = [];
            var $row = $(this);
            
            // Script URL (use full URL from title)
            var scriptUrl = $row.find('.pi-script-url code').attr('title') || $row.find('.pi-script-url code').text();
            row.push('"' + scriptUrl.replace(/"/g, '""') + '"');
            
            // Plugin/Source
            var source = $row.find('.pi-source-type').text();
            var version = $row.find('.pi-version').text();
            if (version) source += ' ' + version;
            row.push('"' + source.replace(/"/g, '""') + '"');
            
            // Issue Type
            row.push('"' + $row.find('.pi-issue-badge').text().replace(/"/g, '""') + '"');
            
            // Impact
            var impact = $row.find('.pi-impact').text().trim();
            row.push('"' + impact.replace(/"/g, '""') + '"');
            
            // Suggestion
            var suggestion = $row.find('.pi-suggestion-text').text().trim();
            row.push('"' + suggestion.replace(/"/g, '""') + '"');
            
            csv.push(row.join(','));
        });
        
        // Download CSV
        var csvContent = csv.join('\n');
        var blob = new Blob([csvContent], { type: 'text/csv;charset=utf-8;' });
        var link = document.createElement('a');
        
        if (link.download !== undefined) {
            var url = URL.createObjectURL(blob);
            link.setAttribute('href', url);
            link.setAttribute('download', 'performance-insights-' + new Date().toISOString().slice(0, 10) + '.csv');
            link.style.visibility = 'hidden';
            document.body.appendChild(link);
            link.click();
            document.body.removeChild(link);
        }
    }
    
    // Keyboard shortcuts
    $(document).on('keydown', function(e) {
        // Ctrl/Cmd + Enter to submit form
        if ((e.ctrlKey || e.metaKey) && e.keyCode === 13) {
            if ($('.pi-form').length) {
                $('.pi-form').submit();
            }
        }
        
        // Escape to clear filters
        if (e.keyCode === 27) {
            $('#pi-filter-source, #pi-filter-severity').val('').trigger('change');
        }
    });
    
    // Add tooltips for severity indicators
    $('.pi-severity-critical').attr('title', 'Critical: High impact on performance - prioritize for exclusion');
    $('.pi-severity-moderate').attr('title', 'Moderate: Medium impact on performance - consider for exclusion');
    $('.pi-severity-fine').attr('title', 'Fine: Low impact on performance - optional exclusion');
    
    // Initialize default sort by impact (descending)
    if ($('.pi-scripts-table').length) {
        $('.pi-sortable[data-sort="impact"]').trigger('click').trigger('click'); // Double click for desc
    }
    
});

// Global object for localized strings (if needed for future localization)
var piAdmin = window.piAdmin || {
    expandText: 'Expand',
    collapseText: 'Collapse'
};
