<?php
if (!defined('ABSPATH')) { exit; }

class PI_Admin {
    public function __construct() {
        add_action('admin_menu', [$this, 'add_admin_menu']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_admin_assets']);
        add_action('admin_post_pi_analyze', [$this, 'handle_analyze']);
    }

    public function add_admin_menu() {
        add_management_page(
            __('Performance Insights', 'performance-insights'),
            __('Performance Insights', 'performance-insights'),
            'manage_options',
            'pi-insights',
            [$this, 'admin_page']
        );
    }

    public function enqueue_admin_assets($hook) {
        if ($hook !== 'tools_page_pi-insights') return;
        
        wp_enqueue_style(
            'pi-admin-style',
            PI_PLUGIN_URL . 'assets/admin.css',
            [],
            '1.0.0'
        );
        
        wp_enqueue_script(
            'pi-admin-script',
            PI_PLUGIN_URL . 'assets/admin.js',
            ['jquery'],
            '1.0.0',
            true
        );
        
        wp_localize_script('pi-admin-script', 'piAdmin', [
            'nonce' => wp_create_nonce('pi_admin_nonce'),
            'ajaxurl' => admin_url('admin-ajax.php')
        ]);
    }

    public function admin_page() {
        $results = null;
        $errors = [];
        
        // Check if we have results from a recent analysis
        if (isset($_GET['analyzed']) && $_GET['analyzed'] === '1') {
            $transient_key = 'pi_results_' . get_current_user_id();
            $stored_results = get_transient($transient_key);
            if ($stored_results) {
                $results = $stored_results;
                delete_transient($transient_key);
            }
        }
        
        include PI_PLUGIN_DIR . 'templates/admin-page.php';
    }

    public function handle_analyze() {
        if (!current_user_can('manage_options')) {
            wp_die(__('You do not have sufficient permissions to access this page.'));
        }
        
        if (!wp_verify_nonce($_POST['pi_nonce'] ?? '', 'pi_analyze')) {
            wp_die(__('Security check failed.'));
        }

        // Get and process the Lighthouse JSON input
        $raw_json = $_POST['lighthouse_json'] ?? '';
        $lighthouse_json = wp_unslash($raw_json);
        
        // If wp_unslash returns null, use raw data
        if ($lighthouse_json === null) {
            $lighthouse_json = $raw_json;
        }
        
        // Ensure we have a string, not null
        if ($lighthouse_json === null) {
            $lighthouse_json = '';
        }
        
        // Skip regex sanitization for large JSON - Lighthouse JSON shouldn't contain script tags
        if (strlen($lighthouse_json) > 500000) { // Skip for files > 500KB
            // Large files - skip sanitization
        } else {
            // Basic sanitization - remove any potential script tags but preserve JSON structure
            $sanitized = preg_replace('/<script\b[^<]*(?:(?!<\/script>)<[^<]*)*<\/script>/mi', '', $lighthouse_json);
            if ($sanitized !== null) {
                $lighthouse_json = $sanitized;
            }
        }
        
        // Parse the JSON with enhanced insights
        $results = PI_Parser::parse($lighthouse_json);
        
        // Add summary insights if no errors and we have scripts
        if (empty($results['errors']) && !empty($results['sections']['scripts']['scripts'])) {
            $results['summary'] = PI_Insights::generate_summary($results['sections']['scripts']['scripts']);
        }
        
        // Store results temporarily
        $transient_key = 'pi_results_' . get_current_user_id();
        set_transient($transient_key, $results, 300); // 5 minutes
        
        // Redirect back with success flag
        $redirect_url = add_query_arg([
            'page' => 'pi-insights',
            'analyzed' => '1'
        ], admin_url('tools.php'));
        
        wp_redirect($redirect_url);
        exit;
    }
}
