<?php
/**
 * Plugin Name: Search and Replace Anywhere
 * Plugin URI: https://wpcraftbench.com/plugins/search-and-replace-anywhere/
 * Description: A lightweight plugin to search for and replace text across your WordPress site, including across Elementor and all other major editors. Features advanced filtering, preview mode, and safe database operations.
 * Version: 1.0
 * Author: WP Craftbench
 * Author URI: https://wpcraftbench.com/
 * License: GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: search-and-replace-anywhere
 * Requires at least: 5.0
 * Tested up to: 6.7
 * Requires PHP: 7.4
 * Network: false
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly.
}

class WP_Search_Replace {

    public function __construct() {
        add_action('admin_menu', array($this, 'register_menu'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_assets'));
        add_action('wp_ajax_wp_search_replace_search', array($this, 'ajax_search'));
        add_action('wp_ajax_wp_search_replace_replace', array($this, 'ajax_replace'));
    }

    /**
     * Register the admin menu.
     */
    public function register_menu() {
        add_menu_page(
            'Search & Replace',
            'Search & Replace',
            'manage_options',
            'wp-search-replace',
            array($this, 'plugin_page'),
            'dashicons-search',
            90
        );
    }

    /**
     * Enqueue CSS and JavaScript.
     */
    public function enqueue_assets($hook) {
        if ($hook !== 'toplevel_page_wp-search-replace') {
            return;
        }

        // Enqueue CSS.
        wp_enqueue_style(
            'wp-search-replace-style',
            plugin_dir_url(__FILE__) . 'css/style.css',
            array(),
            '1.6'
        );

        // Enqueue JS.
        wp_enqueue_script(
            'wp-search-replace-script',
            plugin_dir_url(__FILE__) . 'js/script.js',
            array('jquery'),
            '1.6',
            true
        );

        // Prepare filter data for the custom dropdown.
        $filter_data = array(
            'category'  => array(),
            'post_tag'  => array(),
            'page'      => array(),
            'post'      => array()
        );

        // Categories
        $categories = get_terms(array(
            'taxonomy'   => 'category',
            'hide_empty' => false
        ));
        if (! is_wp_error($categories)) {
            foreach ($categories as $cat) {
                $filter_data['category'][] = array(
                    'id'   => $cat->term_id,
                    'text' => $cat->name
                );
            }
        }

        // Tags (we use key "post_tag" for tags)
        $tags = get_terms(array(
            'taxonomy'   => 'post_tag',
            'hide_empty' => false
        ));
        if (! is_wp_error($tags)) {
            foreach ($tags as $tag) {
                $filter_data['post_tag'][] = array(
                    'id'   => $tag->term_id,
                    'text' => $tag->name
                );
            }
        }

        // Pages
        $pages = get_posts(array(
            'post_type'   => 'page',
            'numberposts' => -1
        ));
        foreach ($pages as $page) {
            $filter_data['page'][] = array(
                'id'   => $page->ID,
                'text' => $page->post_title
            );
        }

        // Posts
        $posts = get_posts(array(
            'post_type'   => 'post',
            'numberposts' => -1
        ));
        foreach ($posts as $post) {
            $filter_data['post'][] = array(
                'id'   => $post->ID,
                'text' => $post->post_title
            );
        }

        // Localize data for the script.
        wp_localize_script('wp-search-replace-script', 'WP_SR', array(
            'ajax_url'    => admin_url('admin-ajax.php'),
            'edit_url'    => admin_url('post.php'),
            'nonce'       => wp_create_nonce('wp_search_replace_nonce'),
            'filter_data' => $filter_data // This should include pages, posts, etc.
        ));
        
    }

    /**
     * Output the admin page interface.
     */
    public function plugin_page() {
        ?>
        <!-- Main container with ID to boost CSS specificity -->
        <div class="wrap wp-sr-container" id="wp-sr-container">
            <!-- Logo-style title -->
            <div class="wp-sr-logo-header">
                <div class="wp-sr-logo">
                    <div class="wp-sr-logo-icon">
                        <span class="dashicons dashicons-search"></span>
                        <span class="dashicons dashicons-randomize"></span>
                    </div>
                    <div class="wp-sr-logo-text">
                        <h1 class="wp-sr-brand-name">Search & Replace</h1>
                        <span class="wp-sr-tagline">Anywhere</span>
                    </div>
                </div>
            </div>
            <hr class="wp-header-end">
            
            <div class="wp-sr-main-content">
            
            <!-- Simple Description -->
            <div class="wp-sr-description">
                <div class="description-text">
                    <p>Search and replace text across your entire WordPress database including all page builders (Elementor, Gutenberg, etc.). Always backup your database before making changes.</p>
                </div>
            </div>
            <div id="wp-sr-interface">
                <div class="wp-sr-search-replace">
                    <div class="wp-sr-search-replace-inner">
                    <form id="wp-sr-form">
                        <div class="wp-sr-form-grid">
                            <div class="form-section">
                                <label for="search_term">
                                    <span class="dashicons dashicons-search" style="margin-right: 8px;"></span>
                                    Search for text:
                                </label>
                                <input id="search_term" name="search_term" required class="wp-sr-input" placeholder="Enter search text...">
                                
                                <div class="checkbox-group">
                                    <label class="checkbox-label">
                                        <input type="checkbox" name="case_sensitive">
                                        <span>Case sensitive</span>
                                    </label>
                                   
                                </div>
                            </div>

                            <div class="form-section">
                                <label for="replace_term">
                                    <span class="dashicons dashicons-randomize" style="margin-right: 8px;"></span>
                                    Replace with text:
                                </label>
                                <input id="replace_term" name="replace_term" class="wp-sr-input" placeholder="Enter replacement text...">
                            </div>
                        </div>

                            <div class="wp-sr-field">
                                <!-- Hidden select for maintaining filter state -->
                                <select id="filter_type" name="filter_type" class="wp-sr-input" style="display:none;">
                                    <option value="all">All</option>
                                    <option value="category">Category</option>
                                    <option value="page">Page</option>
                                    <option value="post">Post</option>
                                    <option value="tag">Tag</option>
                                </select>
                            </div>

                            <div id="filter_values_wrapper" style="display:none;">
                                <div id="filter_values" class="wp-sr-dropdown">
                                    <div class="dropdown-menu" style="display:none;"></div>
                                </div>
                            </div>
                        </div>

                        <div class="button-group">
                           
                            <button type="button" id="wp-sr-search-btn" class="button button-primary">
                                <span class="dashicons dashicons-search"></span>
                                Search
                            </button>
                            <button type="button" id="wp-sr-replace-btn" class="button button-primary" disabled>
                                <span class="dashicons dashicons-randomize"></span>
                                Replace Selected
                                <span class="selected-count">(0)</span>
                            </button>
                        </div>
                    </form>

                    <!-- Progress Bar -->
                    <div id="wp-sr-progress">
                        <div id="wp-sr-progress-bar"></div>
                    </div>
                </div>

                <!-- Results Section -->
                <div class="wp-sr-results">
                    <div class="results-header">
                        <div>
                            <h2 class="results-title">
                                <span class="dashicons dashicons-list-view"></span>
                                Search Results
                            </h2>
                            <p class="results-subtitle">Select items to replace</p>
                        </div>
                        <div class="results-filter">


                            <div id="filter-button" class="wp-sr-input filter-button">
                                <span class="dashicons dashicons-filter"></span>
                                <span>Filter Results</span>
                                <span class="dashicons dashicons-arrow-down"></span>
                            </div>


                            <div class="filter-content" style="display:none;">
                                <div class="filter-sections-container">
                                <div class="filter-section">
                                    <div class="filter-section-title">Content Type</div>
                                    <div id="content-dropdown">
                                        <div class="dropdown-item" data-type="page">Page</div>
                                        <div class="dropdown-item" data-type="post">Post</div>
                                        <div class="dropdown-item" data-type="category">Category</div>
                                        <div class="dropdown-item" data-type="tag">Tag</div>
                                        <div class="dropdown-item" data-type="menu">Menu</div>
                                        <div class="dropdown-item" data-type="widget">Widget</div>
                                    </div>
                                </div>
                                <div id="type-filter-section" class="filter-section" style="display:none;">
                                    <div class="filter-section-type"><div class="filter-search">
                                        <input type="text" 
                                            id="type-filter-search" 
                                            class="wp-search-filter" 
                                            placeholder="Search items..."
                                            style="width: 100%; margin-bottom: 10px;">
                                    </div> </div>
                                   
                                    <div id="type-dropdown"></div>
                                </div>
                                </div>
                            </div>


                        </div>
                    </div>
                    <!-- Selection Controls -->
                    <div class="selection-controls" style="display: none;">
                        <div class="selection-info">
                            <span class="dashicons dashicons-yes-alt"></span>
                            <span id="selection-count">0</span> items selected
                        </div>
                        <div class="selection-actions">
                            <button type="button" id="select-all-btn" class="button button-secondary">
                                <span class="dashicons dashicons-yes"></span>
                                Select All
                            </button>
                            <button type="button" id="select-none-btn" class="button button-secondary">
                                <span class="dashicons dashicons-dismiss"></span>
                                Select None
                            </button>
                            <button type="button" id="select-visible-btn" class="button button-secondary">
                                <span class="dashicons dashicons-visibility"></span>
                                Select Visible
                            </button>
                        </div>
                    </div>

                    <table id="wp-sr-results-table" class="wp-sr-table">
                        <thead>
                            <tr>
                                <th class="select-column">
                                    <input type="checkbox" id="select-all-checkbox" title="Select/Deselect All">
                                </th>
                                <th>Post ID</th>
                                <th>Title</th>
                                <th>Type</th>
                                <th>Snippet (Before)</th>
                                <th>Snippet (After)</th>
                                <th>Matches</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <!-- Populated by AJAX -->
                        </tbody>
                    </table>
                </div>
            </div>
            </div> <!-- Close wp-sr-main-content -->
        </div>
        <?php
    }

    /**
     * Helper function to generate a snippet and highlight matches.
     */
    /**
     * Helper function to generate a snippet and highlight matches.
     * Improved to show better context around matches.
     */
    private function get_snippet($content, $term, $case_sensitive = false, $whole_word = false) {
        if (empty($content) || empty($term)) {
            return '';
        }
        
        // Create the pattern based on options
        $pattern = preg_quote($term, '/');
        if ($whole_word) {
            $pattern = "\\b" . $pattern . "\\b";
        }
        $flags = $case_sensitive ? '' : 'i';
        
        // Find the position of the first match
        preg_match('/' . $pattern . '/' . $flags, $content, $matches, PREG_OFFSET_CAPTURE);
        if (empty($matches)) {
            return '';
        }
        
        $pos = $matches[0][1];
        $snippet = substr($content, $pos);
        
        // Split into words and take only the first 6
        $words = preg_split('/\s+/', $snippet);
        $truncated = array_slice($words, 0, 6);
        $snippet = implode(' ', $truncated);
        
        // Highlight the matched term
        $snippet = preg_replace(
            '/' . $pattern . '/' . $flags,
            '<span class="changed">$0</span>',
            $snippet
        );
        
        return $snippet;
    }

    /**
     * AJAX handler for searching.
     * Optimized for better performance with improved query structure and memory management.
     */
    private function count_matches($content, $term, $case_sensitive = false, $whole_word = false) {
        if (empty($content) || empty($term)) {
            return 0;
        }
        
        $pattern = preg_quote($term, '/');
        if ($whole_word) {
            $pattern = "\\b" . $pattern . "\\b";
        }
        
        $flags = $case_sensitive ? '' : 'i';
        preg_match_all('/' . $pattern . '/' . $flags, $content, $matches);
        return count($matches[0]);
    }
    
    
    
    public function ajax_search() {
        check_ajax_referer('wp_search_replace_nonce', 'nonce');
    
        $search_term = isset($_POST['search_term']) ? sanitize_text_field($_POST['search_term']) : '';
        $case_sensitive = isset($_POST['case_sensitive']) && $_POST['case_sensitive'] === 'true';
        $whole_word = isset($_POST['whole_word']) && $_POST['whole_word'] === 'true';
        $filter_type = isset($_POST['filter_type']) ? sanitize_text_field($_POST['filter_type']) : 'all';
        $filter_value = isset($_POST['filter_value']) ? array_map('intval', (array) $_POST['filter_value']) : array();
    
        if (empty($search_term)) {
            wp_send_json_error('No search term provided.');
        }
    
        global $wpdb;
        // Modify the LIKE clause for whole word matching if needed
        $like = $whole_word ? ' %' . $wpdb->esc_like($search_term) . ' ' : '%' . $wpdb->esc_like($search_term) . '%';
        
        $results = array();
        $seen_ids = array();
    
        try {
            // Search in all content types
            $main_sql = $wpdb->prepare(
                "SELECT DISTINCT SQL_CALC_FOUND_ROWS p.ID, p.post_title, p.post_type, p.post_content
                 FROM $wpdb->posts p
                 LEFT JOIN $wpdb->postmeta pm ON p.ID = pm.post_id
                 WHERE p.post_status = 'publish' 
                 AND p.post_type IN ('page','post')
                 AND (p.post_content LIKE %s OR pm.meta_value LIKE %s)
                 LIMIT 1000",
                $like,
                $like
            );
            
            $main_results = $wpdb->get_results($main_sql);
            
            if ($main_results) {
                foreach ($main_results as $r) {
                    if (!isset($seen_ids[$r->ID])) {
                        $r->snippet_before = $this->get_snippet($r->post_content, $search_term, $case_sensitive, $whole_word);
                        $r->snippet_after = '';
                        
                        // Count matches with options
                        $matches = $this->count_matches($r->post_content, $search_term, $case_sensitive, $whole_word);
                        
                        // Get all meta values and count matches there too
                        $meta_values = $wpdb->get_col($wpdb->prepare(
                            "SELECT meta_value FROM $wpdb->postmeta WHERE post_id = %d",
                            $r->ID
                        ));
                        
                        foreach ($meta_values as $meta_value) {
                            $matches += $this->count_matches($meta_value, $search_term, $case_sensitive, $whole_word);
                        }
                        
                        $r->match_count = $matches;
                        $results[] = $r;
                        $seen_ids[$r->ID] = true;
                    }
                }
            }
    
            // Memory cleanup
            wp_cache_flush();
            
            wp_send_json_success($results);
            
        } catch (Exception $e) {
            wp_send_json_error('Database error: ' . $e->getMessage());
        }
    }

    /**
     * AJAX handler for replacing text.
     * Optimized for better performance and error handling.
     */
    public function ajax_replace() {
        check_ajax_referer('wp_search_replace_nonce', 'nonce');
    
        $search_term = isset($_POST['search_term']) ? sanitize_text_field($_POST['search_term']) : '';
        $replace_term = isset($_POST['replace_term']) ? sanitize_text_field($_POST['replace_term']) : '';
        $filter_type = isset($_POST['filter_type']) ? sanitize_text_field($_POST['filter_type']) : 'all';
        $filter_value = isset($_POST['filter_value']) ? array_map('intval', (array) $_POST['filter_value']) : array();
        $post_ids = isset($_POST['post_ids']) ? array_map('intval', (array) $_POST['post_ids']) : array();
        
        $case_sensitive = isset($_POST['case_sensitive']) && $_POST['case_sensitive'] === 'true';
        $whole_word = isset($_POST['whole_word']) && $_POST['whole_word'] === 'true';
    
        if (empty($search_term)) {
            wp_send_json_error('No search term provided.');
        }
    
        global $wpdb;
        $count = 0;
        $updated_snippets = array();
        $errors = array();
    
        try {
            $wpdb->query('START TRANSACTION');
    
            // Only proceed with posts that match our filter criteria
            $where_conditions = array("post_status = 'publish'");
            
            // Add post type condition if filtering by type
            if ($filter_type === 'page' || $filter_type === 'post') {
                $where_conditions[] = $wpdb->prepare("post_type = %s", $filter_type);
            }
            
            // Add specific post IDs condition if we have them
            if (!empty($post_ids)) {
                $post_ids_string = implode(',', array_map('intval', $post_ids));
                $where_conditions[] = "ID IN ($post_ids_string)";
            }
    
            // Build the query
            $posts_sql = $wpdb->prepare(
                "SELECT ID, post_content
                 FROM $wpdb->posts
                 WHERE " . implode(' AND ', $where_conditions),
                $like
            );
    
            $posts = $wpdb->get_results($posts_sql);
            foreach ($posts as $post) {
                // Use regex for whole word matching if needed
                if ($whole_word) {
                    $pattern = '/\b' . preg_quote($search_term, '/') . '\b/';
                    $pattern .= $case_sensitive ? '' : 'i';
                    $new_content = preg_replace($pattern, $replace_term, $post->post_content);
                } else {
                    // Use str functions for simple replacements
                    $new_content = $case_sensitive 
                        ? str_replace($search_term, $replace_term, $post->post_content)
                        : str_ireplace($search_term, $replace_term, $post->post_content);
                }
    
                if ($new_content !== $post->post_content) {
                    $result = wp_update_post(array(
                        'ID' => $post->ID,
                        'post_content' => $new_content
                    ), true);
    
                    if (is_wp_error($result)) {
                        $errors[] = sprintf('Failed to update post %d: %s', $post->ID, $result->get_error_message());
                        continue;
                    }
    
                    $count++;
                    clean_post_cache($post->ID);
                    $updated_snippets[$post->ID] = $this->get_snippet($new_content, $replace_term, $case_sensitive, $whole_word);
                }
            }
    
            $wpdb->query('COMMIT');
            wp_cache_flush();
    
            $response = array(
                'message' => sprintf('%d replacements made.', $count),
                'updated_snippets' => $updated_snippets
            );
    
            if (!empty($errors)) {
                $response['warnings'] = $errors;
            }
    
            wp_send_json_success($response);
    
        } catch (Exception $e) {
            $wpdb->query('ROLLBACK');
            wp_send_json_error('Database error: ' . $e->getMessage());
        }
    }
    
}

new WP_Search_Replace();
