jQuery(document).ready(function ($) {
    // Store search results globally for client-side filtering
    let allResults = [];
    let selectedRows = new Set(); // Track selected row IDs
  
    // Debounce function to limit the rate at which a function can fire
    function debounce(func, wait) {
      let timeout;
      return function executedFunction(...args) {
        const later = () => {
          clearTimeout(timeout);
          func(...args);
        };
        clearTimeout(timeout);
        timeout = setTimeout(later, wait);
      };
    }
    
    // Update selection counter and button state
    function updateSelectionUI() {
        const count = selectedRows.size;
        $('#selection-count').text(count);
        $('.selected-count').text(`(${count})`);
        
        // Enable/disable replace button based on selection
        $('#wp-sr-replace-btn').prop('disabled', count === 0);
        
        // Show/hide selection controls
        if (allResults.length > 0) {
            $('.selection-controls').show();
        } else {
            $('.selection-controls').hide();
        }
        
        // Update master checkbox state
        const visibleRows = $('#wp-sr-results-table tbody tr:visible').length;
        const selectedVisibleRows = $('#wp-sr-results-table tbody tr:visible.selected').length;
        
        if (selectedVisibleRows === 0) {
            $('#select-all-checkbox').prop('indeterminate', false).prop('checked', false);
        } else if (selectedVisibleRows === visibleRows) {
            $('#select-all-checkbox').prop('indeterminate', false).prop('checked', true);
        } else {
            $('#select-all-checkbox').prop('indeterminate', true);
        }
    }
  
    // Helper: generate the edit post link.
    function get_edit_post_link(postID) {
      return WP_SR.edit_url + "?post=" + postID + "&action=edit";
    }
  
    // Filter dropdown handlers
    $("#filter-button").on("click", function (e) {
      e.stopPropagation();
      const $this = $(this);
      const $content = $(".filter-content");
      const isVisible = $content.is(":visible");
  
      $this.toggleClass("active");
      $content.fadeToggle(150);
  
      $this
        .find(".dashicons-arrow-down")
        .css("transform", isVisible ? "rotate(0deg)" : "rotate(180deg)");
    });
  
    // Hide the filter options if clicking outside the filter menu
    $(document).on("click", function (e) {
      if (!$(e.target).closest(".results-filter").length) {
        $(".filter-content").fadeOut(150);
        $("#filter-button")
          .removeClass("active")
          .find(".dashicons-arrow-down")
          .css("transform", "rotate(0deg)");
      }
    });
  
// Store all filter items for searching
let currentFilterItems = [];
let selectedItems = new Set(); // Keep track of selected items globally

// Update the content type selection handler
$("#content-dropdown").on("click", ".dropdown-item", function () {
    const $this = $(this);
    const filterType = $this.data("type");

    // Deselect all other content filters
    $("#content-dropdown .dropdown-item").not($this).removeClass("selected");
    $this.toggleClass("selected");

    const isSelected = $this.hasClass("selected");

    if (!isSelected) {
        $("#type-filter-section").fadeOut(150);
        $("#type-dropdown").empty();
        currentFilterItems = []; // Clear stored items
        selectedItems.clear(); // Clear selected items
        return;
    }

    $("#filter_type").val(filterType);
    const filterKey = filterType === "tag" ? "post_tag" : filterType;
    const filterItems = WP_SR.filter_data[filterKey];

    if (filterItems && filterItems.length > 0) {
        // Store the current items
        currentFilterItems = filterItems;
        
        // Update dropdown with current items and preserve selections
        updateTypeDropdown(filterItems);
        
        // Clear search input when changing type
        $("#type-filter-search").val("");
        
        $("#type-filter-section").fadeIn(150);
    }

    filterAndRenderResults();
});

// Handle search input - use single event handler
$("#type-filter-search").on("input", function() {
    const searchTerm = $(this).val().toLowerCase().trim();
    
    // Filter the items based on search term
    const filteredItems = currentFilterItems.filter(item => 
        item.text.toLowerCase().includes(searchTerm)
    );
    
    // Update dropdown with filtered items while preserving selections
    updateTypeDropdown(filteredItems);
});

// Handle item selection in dropdown
$("#type-dropdown").on("click", ".dropdown-item", function() {
    const $item = $(this);
    const itemId = $item.data("id");
    
    if ($item.hasClass("selected")) {
        selectedItems.delete(itemId);
        $item.removeClass("selected");
    } else {
        selectedItems.add(itemId);
        $item.addClass("selected");
    }
    
    filterAndRenderResults();
});

// Helper function to update dropdown while preserving selections
function updateTypeDropdown(items) {
    const $typeDropdown = $("#type-dropdown");
    
    if (items.length) {
        const html = items
            .map((item) => {
                const isSelected = selectedItems.has(item.id);
                return `<div class="dropdown-item${isSelected ? ' selected' : ''}" 
                            data-id="${item.id}">${item.text}</div>`;
            })
            .join("");
        
        $typeDropdown.html(html);
    } else {
        $typeDropdown.html('<div class="no-results">No matching items found</div>');
    }
}
  






    // Optimized render function with DOM fragment and requestAnimationFrame
   // Optimized render function with DOM fragment and requestAnimationFrame
   function renderResults(results) {
    const searchText = $("#search_term").val();
    const replaceText = $("#replace_term").val();
    const tbody = document.querySelector("#wp-sr-results-table tbody");
    
    const fragment = document.createDocumentFragment();
    
    results.forEach((item) => {
        const row = document.createElement("tr");
        row.dataset.postId = item.ID;
        
        // Add selected class if this row is selected
        if (selectedRows.has(item.ID)) {
            row.classList.add('selected');
        }
        
        const title = item.post_type === "page" 
            ? `<strong>${item.post_title}</strong>` 
            : item.post_title;
        
        const beforeSnippetHtml = generatePreviewSnippet(
            item.snippet_before,
            searchText,
            replaceText,
            false
        );
        
        const afterSnippetHtml = replaceText
            ? generatePreviewSnippet(item.snippet_before, searchText, replaceText, true)
            : "";
        
        // Use the match_count from the backend instead of calculating it here
        const matches = item.match_count || 0;
        
        const isChecked = selectedRows.has(item.ID) ? 'checked' : '';
        
        row.innerHTML = `
            <td class="select-column">
                <input type="checkbox" class="row-select" data-post-id="${item.ID}" ${isChecked}>
            </td>
            <td>${item.ID}</td>
            <td>${title}</td>
            <td>${item.post_type}</td>
            <td class="snippet-before">${beforeSnippetHtml}</td>
            <td class="snippet-after">${afterSnippetHtml}</td>
            <td class="matches">${matches}</td>
            <td><a href="${get_edit_post_link(item.ID)}" target="_blank">Edit</a></td>
        `;
        
        fragment.appendChild(row);
    });
    
    requestAnimationFrame(() => {
        tbody.innerHTML = "";
        tbody.appendChild(fragment);
        updateSelectionUI();
    });
}
  

  

    

const generatePreviewSnippet = (() => {
    const cache = new Map();
    const maxCacheSize = 100;

    return (beforeSnippet, searchText, replaceText, after = false) => {
        if (!beforeSnippet || !searchText) return "";

        const case_sensitive = $("input[name='case_sensitive']").prop('checked');
        const whole_word = $("input[name='whole_word']").prop('checked');

        const key = `${beforeSnippet}|${searchText}|${replaceText}|${after}|${case_sensitive}|${whole_word}`;
        if (cache.has(key)) return cache.get(key);

        const div = document.createElement("div");
        div.innerHTML = beforeSnippet;
        const text = div.textContent;

        // Create the appropriate regex based on options
        let regexPattern = searchText.replace(/[.*+?^${}()|[\]\\]/g, "\\$&");
        if (whole_word) {
            regexPattern = `\\b${regexPattern}\\b`;
        }
        const flags = case_sensitive ? "g" : "gi";
        const regex = new RegExp(regexPattern, flags);

        let result = "";

        if (after && replaceText) {
            const replacedText = text.replace(regex, replaceText);
            const replaceIndex = case_sensitive 
                ? replacedText.indexOf(replaceText)
                : replacedText.toLowerCase().indexOf(replaceText.toLowerCase());

            if (replaceIndex === -1) return "";

            const relevantText = replacedText.slice(replaceIndex);
            const wordList = relevantText.split(/\s+/).filter(word => word && word.trim().length > 0);
            const finalSnippet = wordList.slice(0, 6).join(" ");

            // Create highlight regex for replacement text
            let highlightPattern = replaceText.replace(/[.*+?^${}()|[\]\\]/g, "\\$&");
            if (whole_word) {
                highlightPattern = `^${highlightPattern}\\b`;
            }
            const highlightRegex = new RegExp(highlightPattern, case_sensitive ? "" : "i");
            result = finalSnippet.replace(highlightRegex, match => `<span class="changed">${match}</span>`);
        } else {
            const matchIndex = case_sensitive 
                ? text.indexOf(searchText)
                : text.toLowerCase().indexOf(searchText.toLowerCase());

            if (matchIndex === -1) return "";

            const relevantText = text.slice(matchIndex);
            const wordList = relevantText.split(/\s+/).filter(word => word && word.trim().length > 0);
            const finalSnippet = wordList.slice(0, 6).join(" ");

            result = finalSnippet.replace(regex, match => `<span class="changed">${match}</span>`);
        }

        if (cache.size >= maxCacheSize) {
            const firstKey = cache.keys().next().value;
            cache.delete(firstKey);
        }
        cache.set(key, result);

        return result;
    };
})();
  
  
    

  
    // Function to apply filters and render results
    function filterAndRenderResults() {
      const $selectedType = $('#content-dropdown .dropdown-item.selected');
      const $selectedItems = $('#type-dropdown .dropdown-item.selected');
  
      let filteredResults = [...allResults];
  
      // Filter by content type (only one can be selected)
      if ($selectedType.length) {
        const type = $selectedType.data("type");
        filteredResults = filteredResults.filter((item) => {
          switch (type) {
            case "page":
              return item.post_type === "page";
            case "post":
              return item.post_type === "post";
            case "category":
              return item.taxonomy === "category";
            case "tag":
              return item.taxonomy === "post_tag";
            default:
              return true;
          }
        });
      }
  
      // Filter by selected items (multiple can be selected)
      if ($selectedItems.length) {
        const selectedIds = new Set(
          Array.from($selectedItems).map((item) => Number($(item).data("id")))
        );
  
        filteredResults = filteredResults.filter((item) => {
          const id = Number(item.ID);
          return selectedIds.has(id);
        });
      }
  
      renderResults(filteredResults);
    }

    // Debounced version of filter and render
    const debouncedFilterAndRender = debounce(filterAndRenderResults, 150);

    // Selection event handlers
    
    // Master checkbox handler
    $('#select-all-checkbox').on('change', function() {
        const isChecked = $(this).prop('checked');
        $('#wp-sr-results-table tbody tr:visible').each(function() {
            const postId = parseInt($(this).data('postId')); // Fixed: use postId instead of post-id
            const checkbox = $(this).find('.row-select');
            
            if (isChecked) {
                selectedRows.add(postId);
                $(this).addClass('selected');
                checkbox.prop('checked', true);
            } else {
                selectedRows.delete(postId);
                $(this).removeClass('selected');
                checkbox.prop('checked', false);
            }
        });
        updateSelectionUI();
    });
    
    // Individual row checkbox handler
    $(document).on('change', '.row-select', function() {
        const postId = parseInt($(this).data('post-id'));
        const row = $(this).closest('tr');
        
        if ($(this).prop('checked')) {
            selectedRows.add(postId);
            row.addClass('selected');
        } else {
            selectedRows.delete(postId);
            row.removeClass('selected');
        }
        updateSelectionUI();
    });
    
    // Row click handler (click anywhere on row to select)
    $(document).on('click', '#wp-sr-results-table tbody tr', function(e) {
        // Don't trigger if clicking on checkbox, link, or other interactive elements
        if ($(e.target).is('input, a, button') || $(e.target).closest('a, button').length) {
            return;
        }
        
        const checkbox = $(this).find('.row-select');
        checkbox.prop('checked', !checkbox.prop('checked')).trigger('change');
    });
    
    // Selection control buttons
    $('#select-all-btn').on('click', function() {
        $('#select-all-checkbox').prop('checked', true).trigger('change');
    });
    
    $('#select-none-btn').on('click', function() {
        $('#select-all-checkbox').prop('checked', false).trigger('change');
    });
    
    $('#select-visible-btn').on('click', function() {
        $('#wp-sr-results-table tbody tr:visible').each(function() {
            const postId = parseInt($(this).data('postId')); // Fixed: use postId instead of post-id
            const checkbox = $(this).find('.row-select');
            
            selectedRows.add(postId);
            $(this).addClass('selected');
            checkbox.prop('checked', true);
        });
        updateSelectionUI();
    });

// Handle Search button click
$("#wp-sr-search-btn").on("click", function () {
    const search_term = $("#search_term").val().trim();
    const case_sensitive = $("input[name='case_sensitive']").prop('checked');
    const whole_word = $("input[name='whole_word']").prop('checked');
    
    if (!search_term) {
        alert("Please enter a search term");
        return;
    }

    // Clear previous results and reset filters
    allResults = [];
    selectedRows.clear(); // Clear selections
    $("#type-dropdown").empty();
    $("#type-filter-section").hide();
    $('#content-dropdown .dropdown-item').removeClass("selected");

    // Clear results table
    $("#wp-sr-results-table tbody").html('<tr><td colspan="8" style="text-align: center; padding: 2rem;"><div class="wp-sr-loading"><span class="dashicons dashicons-update-alt" style="animation: spin 1s linear infinite; font-size: 24px; color: #4f46e5;"></span><br><strong>Searching...</strong><br><small>Please wait while we search your content</small></div></td></tr>');

    // Disable search button and show loading state
    const $searchBtn = $(this);
    const originalText = $searchBtn.html();
    $searchBtn.prop('disabled', true).html('<span class="dashicons dashicons-update-alt" style="animation: spin 1s linear infinite;"></span> Searching...');

    const $progress = $("#wp-sr-progress");
    const $progressBar = $("#wp-sr-progress-bar");

    $progress.show();
    $progressBar.css("width", "0%").animate({ width: "70%" }, 1000);

    $.ajax({
        url: WP_SR.ajax_url,
        type: "POST",
        dataType: "json",
        data: {
            action: "wp_search_replace_search",
            nonce: WP_SR.nonce,
            search_term: search_term,
            case_sensitive: case_sensitive,
            whole_word: whole_word,
            filter_type: "all",
            filter_value: [],
        },
        success: function (response) {
          if (response.success) {
            // Cache the results globally
            allResults = response.data;
            // Apply any existing filters
            filterAndRenderResults();
            
            // Show success message if results found
            if (response.data.length > 0) {
                const resultCount = response.data.length;
                const totalMatches = response.data.reduce((sum, item) => sum + (item.match_count || 0), 0);
                console.log(`Search completed: ${resultCount} items found with ${totalMatches} total matches`);
                
                // Auto-select all results for now to enable replace button
                setTimeout(() => {
                    response.data.forEach(item => {
                        selectedRows.add(parseInt(item.ID));
                    });
                    updateSelectionUI();
                    
                    // Update checkboxes
                    $('#wp-sr-results-table tbody tr').each(function() {
                        const postId = parseInt($(this).data('postId'));
                        if (selectedRows.has(postId)) {
                            $(this).addClass('selected');
                            $(this).find('.row-select').prop('checked', true);
                        }
                    });
                }, 100);
            } else {
                $("#wp-sr-results-table tbody").html('<tr><td colspan="8" style="text-align: center; padding: 2rem; color: #6b7280;"><span class="dashicons dashicons-search" style="font-size: 48px; opacity: 0.3;"></span><br><strong>No results found</strong><br><small>Try adjusting your search term or filters</small></td></tr>');
            }
            
            $("#wp-sr-progress-bar").animate({ width: "100%" }, 500, function () {
              setTimeout(function () {
                $("#wp-sr-progress-bar").css("width", "0%");
                $("#wp-sr-progress").hide();
              }, 500);
            });
          } else {
            alert(response.data);
            $("#wp-sr-results-table tbody").html('<tr><td colspan="8" style="text-align: center; padding: 2rem; color: #dc2626;"><span class="dashicons dashicons-warning" style="font-size: 48px; opacity: 0.5;"></span><br><strong>Search Error</strong><br><small>' + response.data + '</small></td></tr>');
            $("#wp-sr-progress-bar").css("width", "0%");
            $("#wp-sr-progress").hide();
          }
        },
        error: function (xhr, status, error) {
          alert("An error occurred during search: " + error);
          $("#wp-sr-results-table tbody").html('<tr><td colspan="8" style="text-align: center; padding: 2rem; color: #dc2626;"><span class="dashicons dashicons-warning" style="font-size: 48px; opacity: 0.5;"></span><br><strong>Connection Error</strong><br><small>Please try again or contact support</small></td></tr>');
          $("#wp-sr-progress-bar").css("width", "0%");
          $("#wp-sr-progress").hide();
        },
        complete: function() {
            // Re-enable search button and restore original text
            $searchBtn.prop('disabled', false).html(originalText);
        }
      });
    });
  
    // Optimize replace button handler
    $("#wp-sr-replace-btn").on("click", function () {
        const search_term = $("#search_term").val().trim();
        const replace_term = $("#replace_term").val().trim();
        const case_sensitive = $("input[name='case_sensitive']").prop('checked');
        const whole_word = $("input[name='whole_word']").prop('checked');
        
        // Get the selected filter type
        const filterType = $('#content-dropdown .dropdown-item.selected').data('type') || 'all';
        
        if (!search_term) {
            alert("Please enter a search term");
            return;
        }
    
        // Get post IDs from selected rows
        const post_ids = Array.from(selectedRows);
        
        if (post_ids.length === 0) {
            alert("Please select at least one item to replace.");
            return;
        }
        
        // Show confirmation dialog with details
        const itemCount = post_ids.length;
        const confirmMessage = `Are you sure you want to replace "${search_term}" with "${replace_term}" in ${itemCount} item(s)?\n\nThis action cannot be undone. Make sure you have a backup of your database.`;
        
        if (!confirm(confirmMessage)) {
            return;
        }
        
        // Disable replace button and show loading state
        const $replaceBtn = $(this);
        const originalText = $replaceBtn.html();
        $replaceBtn.prop('disabled', true).html('<span class="dashicons dashicons-update-alt" style="animation: spin 1s linear infinite;"></span> Replacing...');
        
        // Also disable search button during replace
        $("#wp-sr-search-btn").prop('disabled', true);
    
        const $progress = $("#wp-sr-progress");
        const $progressBar = $("#wp-sr-progress-bar");
    
        $progress.show();
        $progressBar.css("width", "0%").animate({ width: "70%" }, 1000);
    
        $.ajax({
            url: WP_SR.ajax_url,
            type: "POST",
            dataType: "json",
            data: {
                action: "wp_search_replace_replace",
                nonce: WP_SR.nonce,
                search_term,
                replace_term,
                case_sensitive,
                whole_word,
                filter_type: filterType,
                filter_value: Array.from(selectedItems).map(item => $(item).data('id')),
                post_ids: post_ids
            },
            success: function (response) {
                if (response.success) {
                    // Show detailed success message
                    const message = response.data.message;
                    alert("✅ " + message + "\n\nThe page will refresh to show updated results.");
                    
                    const updated = response.data.updated_snippets;
                    const tbody = $("#wp-sr-results-table tbody")[0];
                    Array.from(tbody.getElementsByTagName("tr")).forEach((row) => {
                        const postId = row.dataset.postId;
                        if (updated[postId]) {
                            row.querySelector(".snippet-after").innerHTML = updated[postId];
                            // Add visual indicator that this row was updated
                            $(row).addClass('updated-row').css('background-color', 'rgba(34, 197, 94, 0.1)');
                        }
                    });
                    
                    // Show warnings if any
                    if (response.data.warnings && response.data.warnings.length > 0) {
                        console.warn("Replace operation warnings:", response.data.warnings);
                        alert("⚠️ Operation completed with warnings. Check browser console for details.");
                    }
                    
                    // Optionally refresh search results to show current state
                    setTimeout(() => {
                        $("#wp-sr-search-btn").trigger('click');
                    }, 2000);
                    
                } else {
                    alert("❌ Replace operation failed: " + response.data);
                }
            },
            error: function (xhr, status, error) {
                alert("❌ An error occurred during replace operation: " + error + "\n\nPlease check your database and try again.");
                console.error("Replace error:", xhr.responseText);
            },
            complete: function () {
                $progressBar.animate({ width: "100%" }, 500, () => {
                    setTimeout(() => {
                        $progressBar.css("width", "0%");
                        $progress.hide();
                        
                        // Re-enable buttons
                        $replaceBtn.prop('disabled', false).html(originalText);
                        $("#wp-sr-search-btn").prop('disabled', false);
                    }, 500);
                });
            },
        });
    });
  });
  