<?php
/**
 * Conditions management class
 */

if (!defined('ABSPATH')) {
    exit;
}

class PAGESPSC_Conditions {
    private $table_name;
    
    public function __construct() {
        global $wpdb;
        $this->table_name = $wpdb->prefix . 'pagespsc_conditions';
    }
    
    public function get_active_conditions() {
        global $wpdb;
        
        $cache_key = 'pagespsc_active_conditions';
        $result = wp_cache_get($cache_key);
        
        if (false === $result) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table requires direct query
            $result = $wpdb->get_results($wpdb->prepare("SELECT * FROM `{$wpdb->prefix}pagespsc_conditions` WHERE status = %s", 'active'));
            wp_cache_set($cache_key, $result, '', 300); // Cache for 5 minutes
        }
        
        return $result;
    }
    
    public function get_all_conditions() {
        global $wpdb;
        
        $cache_key = 'pagespsc_all_conditions';
        $result = wp_cache_get($cache_key);
        
        if (false === $result) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table requires direct query
            $result = $wpdb->get_results("SELECT * FROM `{$wpdb->prefix}pagespsc_conditions` ORDER BY created_at DESC");
            wp_cache_set($cache_key, $result, '', 300); // Cache for 5 minutes
        }
        
        return $result;
    }
    
    public function get_condition($id) {
        global $wpdb;
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table requires direct query, single row lookup
        $result = $wpdb->get_row(
            $wpdb->prepare(
                "SELECT * FROM `{$wpdb->prefix}pagespsc_conditions` WHERE id = %d",
                $id
            )
        );
        
        return $result;
    }
    
    public function create_condition($data) {
        global $wpdb;
        
        
        // Validate required fields
        if (empty($data['condition_name'])) {
            // Missing condition_name
            return new WP_Error('missing_name', 'Condition name is required');
        }
        
        // Prepare pages metadata
        $pages_meta = array();
        if (!empty($data['pages'])) {
            foreach ($data['pages'] as $page_id) {
                $page = get_post($page_id);
                if ($page) {
                    $page_url = str_replace(home_url(), '', get_permalink($page_id));
                    $pages_meta[] = array(
                        'id' => $page_id,
                        'slug' => $page->post_name,
                        'url' => trim($page_url, '/') // Remove leading/trailing slashes for consistent matching
                    );
                }
            }
        }
        
        $insert_data = array(
            'condition_name' => sanitize_text_field($data['condition_name']),
            'plugins' => json_encode(isset($data['plugins']) ? $data['plugins'] : array()),
            'include_exclude' => sanitize_text_field(isset($data['include_exclude']) ? $data['include_exclude'] : 'include'),
            'pages' => json_encode(isset($data['pages']) ? $data['pages'] : array()),
            'pages_meta' => json_encode($pages_meta),
            'categories' => json_encode(isset($data['categories']) ? $data['categories'] : array()),
            'tags' => json_encode(isset($data['tags']) ? $data['tags'] : array()),
            'status' => 'active'
        );
        
        // Creating new condition
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table requires direct query
        $result = $wpdb->insert(
            $wpdb->prefix . 'pagespsc_conditions',
            $insert_data,
            array('%s', '%s', '%s', '%s', '%s', '%s', '%s', '%s')
        );
        
        if ($result === false) {
            // Database error occurred
            return new WP_Error('db_error', 'Failed to create condition: ' . $wpdb->last_error);
        }
        
        // Clear cache after creating condition
        wp_cache_delete('pagespsc_all_conditions');
        wp_cache_delete('pagespsc_active_conditions');
        wp_cache_delete('pagespsc_conditions_count');
        
        // Condition created successfully
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table requires direct query, single row lookup
        $result = $wpdb->get_row(
            $wpdb->prepare(
                "SELECT * FROM `{$wpdb->prefix}pagespsc_conditions` WHERE id = %d",
                $wpdb->insert_id
            )
        );
        
        return $result;
    }
    
    public function update_condition($id, $data) {
        global $wpdb;
        
        
        $update_data = array();
        $update_format = array();
        
        if (isset($data['condition_name'])) {
            $update_data['condition_name'] = sanitize_text_field($data['condition_name']);
            $update_format[] = '%s';
        }
        
        if (isset($data['plugins'])) {
            $update_data['plugins'] = json_encode($data['plugins']);
            $update_format[] = '%s';
        }
        
        if (isset($data['include_exclude'])) {
            $update_data['include_exclude'] = sanitize_text_field($data['include_exclude']);
            $update_format[] = '%s';
        }
        
        if (isset($data['pages'])) {
            $update_data['pages'] = json_encode($data['pages']);
            $update_format[] = '%s';
            
            // Update pages metadata
            $pages_meta = array();
            if (!empty($data['pages'])) {
                foreach ($data['pages'] as $page_id) {
                    $page = get_post($page_id);
                    if ($page) {
                        $pages_meta[] = array(
                            'id' => $page_id,
                            'slug' => $page->post_name,
                            'url' => str_replace(home_url(), '', get_permalink($page_id))
                        );
                    }
                }
            }
            $update_data['pages_meta'] = json_encode($pages_meta);
            $update_format[] = '%s';
        }
        
        if (isset($data['categories'])) {
            $update_data['categories'] = json_encode($data['categories']);
            $update_format[] = '%s';
        }
        
        if (isset($data['tags'])) {
            $update_data['tags'] = json_encode($data['tags']);
            $update_format[] = '%s';
        }
        
        if (isset($data['status'])) {
            $update_data['status'] = sanitize_text_field($data['status']);
            $update_format[] = '%s';
        }
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table requires direct query
        $result = $wpdb->update(
            $wpdb->prefix . 'pagespsc_conditions',
            $update_data,
            array('id' => $id),
            $update_format,
            array('%d')
        );
        
        if ($result === false) {
            // Database update error
            return new WP_Error('db_error', 'Failed to update condition: ' . $wpdb->last_error);
        }
        
        // Clear cache after updating condition
        wp_cache_delete('pagespsc_all_conditions');
        wp_cache_delete('pagespsc_active_conditions');
        wp_cache_delete('pagespsc_conditions_count');
        
        // Condition updated successfully
        return true;
    }
    
    public function delete_condition($id) {
        global $wpdb;
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table requires direct query
        $result = $wpdb->delete($wpdb->prefix . 'pagespsc_conditions', array('id' => $id), array('%d'));
        
        if ($result === false) {
            return new WP_Error('db_error', 'Failed to delete condition');
        }
        
        // Clear cache after deleting condition
        wp_cache_delete('pagespsc_all_conditions');
        wp_cache_delete('pagespsc_active_conditions');
        wp_cache_delete('pagespsc_conditions_count');
        
        return true;
    }
    
    // REST API callbacks
    public function get_conditions_api() {
        $conditions = $this->get_all_conditions();
        
        // Decode JSON fields
        foreach ($conditions as &$condition) {
            $condition->plugins = json_decode($condition->plugins, true);
            $condition->pages = json_decode($condition->pages, true);
            $condition->categories = json_decode($condition->categories, true);
            $condition->tags = json_decode($condition->tags, true);
        }
        
        return rest_ensure_response($conditions);
    }
    
    public function create_condition_api($request) {
        $params = $request->get_json_params();
        
        // If no JSON params, try getting from body params
        if (empty($params)) {
            $params = $request->get_body_params();
        }
        
        // If still empty, try getting from regular params (for AJAX compatibility)
        if (empty($params)) {
            $params = $request->get_params();
        }
        
        // Processing API create request
        
        // Validate required fields
        if (empty($params['condition_name'])) {
            return new WP_Error('missing_name', 'Condition name is required', array('status' => 400));
        }

        // Freemium enforcement: limit total conditions and block advanced targeting on free plan
        $is_premium = function_exists('pagespsc_is_premium') ? pagespsc_is_premium() : false;
        if (!$is_premium) {
            $remaining = class_exists('PAGESPSC_Freemium') ? PAGESPSC_Freemium::get_remaining_conditions() : 0;
            if ($remaining <= 0) {
                return new WP_Error(
                    'limit_reached',
                    'Free plan limit reached: Upgrade to create more conditions.',
                    array(
                        'status' => 403,
                        'upgrade_url' => function_exists('pagespsc_get_upgrade_url') ? pagespsc_get_upgrade_url() : ''
                    )
                );
            }
            // Disallow categories/tags on free plan
            if (!empty($params['categories']) || !empty($params['tags'])) {
                return new WP_Error(
                    'premium_feature',
                    'Categories and tags targeting are Pro features. Please remove them or upgrade.',
                    array(
                        'status' => 403,
                        'upgrade_url' => function_exists('pagespsc_get_upgrade_url') ? pagespsc_get_upgrade_url() : ''
                    )
                );
            }
        }
        
        $data = array(
            'condition_name' => $params['condition_name'],
            'plugins' => isset($params['plugins']) ? $params['plugins'] : array(),
            'include_exclude' => isset($params['include_exclude']) ? $params['include_exclude'] : 'include',
            'pages' => isset($params['pages']) ? $params['pages'] : array(),
            'categories' => (!$is_premium ? array() : (isset($params['categories']) ? $params['categories'] : array())),
            'tags' => (!$is_premium ? array() : (isset($params['tags']) ? $params['tags'] : array()))
        );
        
        $result = $this->create_condition($data);
        
        if (is_wp_error($result)) {
            return $result;
        }
        
        $condition = $this->get_condition($result);
        if (!$condition) {
            return new WP_Error('creation_failed', 'Failed to create condition', array('status' => 500));
        }
        
        $condition->plugins = json_decode($condition->plugins, true) ?: array();
        $condition->pages = json_decode($condition->pages, true) ?: array();
        $condition->categories = json_decode($condition->categories, true) ?: array();
        $condition->tags = json_decode($condition->tags, true) ?: array();
        
        return rest_ensure_response($condition);
    }
    
    public function update_condition_api($request) {
        $id = (int) $request->get_param('id');
        $params = $request->get_json_params();
        
        // If no JSON params, try getting from body params
        if (empty($params)) {
            $params = $request->get_body_params();
        }
        
        // If still empty, try getting from regular params (for AJAX compatibility)
        if (empty($params)) {
            $params = $request->get_params();
        }
        
        // Processing API update request
        
        // Remove id from params if it exists
        unset($params['id']);
        
        // Ensure we have valid data
        if (empty($params)) {
            return new WP_Error('no_data', 'No data provided for update', array('status' => 400));
        }

        // Freemium enforcement on update for advanced targeting fields
        $is_premium = function_exists('pagespsc_is_premium') ? pagespsc_is_premium() : false;
        if (!$is_premium) {
            if (!empty($params['categories']) || !empty($params['tags'])) {
                return new WP_Error(
                    'premium_feature',
                    'Categories and tags targeting are Pro features. Please remove them or upgrade.',
                    array(
                        'status' => 403,
                        'upgrade_url' => function_exists('pagespsc_get_upgrade_url') ? pagespsc_get_upgrade_url() : ''
                    )
                );
            }
        }
        
        $result = $this->update_condition($id, $params);
        
        if (is_wp_error($result)) {
            return $result;
        }
        
        $condition = $this->get_condition($id);
        if (!$condition) {
            return new WP_Error('not_found', 'Condition not found', array('status' => 404));
        }
        
        $condition->plugins = json_decode($condition->plugins, true) ?: array();
        $condition->pages = json_decode($condition->pages, true) ?: array();
        $condition->categories = json_decode($condition->categories, true) ?: array();
        $condition->tags = json_decode($condition->tags, true) ?: array();
        
        return rest_ensure_response($condition);
    }
    
    public function delete_condition_api($request) {
        // Check if the request is valid
        if (empty($request)) {
            return new WP_Error('invalid_request', 'Invalid request object', array('status' => 400));
        }
        
        // Get the ID parameter
        $id = (int) $request->get_param('id');
        
        // Verify the ID is valid
        if (empty($id)) {
            return new WP_Error('invalid_id', 'Invalid condition ID', array('status' => 400));
        }
        
        // Check if the condition exists before attempting to delete
        $existing = $this->get_condition($id);
        if (!$existing) {
            return new WP_Error('not_found', 'Condition not found', array('status' => 404));
        }
        
        // Perform the deletion
        $result = $this->delete_condition($id);
        
        // Handle the result
        if (is_wp_error($result)) {
            return $result;
        }
        
        if ($result === false) {
            return new WP_Error('deletion_failed', 'Failed to delete condition', array('status' => 500));
        }
        
        // Return a success response
        $response = new WP_REST_Response(array(
            'success' => true,
            'message' => 'Condition deleted successfully',
            'id' => $id
        ));
        
        // Set the status code to 200 (OK) since the deletion was successful
        $response->set_status(200);
        
        return $response;
    }
}
