const { createElement, useState, useEffect, Fragment } = window.wp.element;
const { 
  Button, 
  TextControl, 
  SelectControl,
  CheckboxControl,
  Notice,
  Spinner,
  Modal,
  __experimentalVStack: VStack,
  TabPanel,
  SearchControl
} = window.wp.components;
const apiFetch = window.wp.apiFetch;

// Configure apiFetch with proper headers
apiFetch.use((options, next) => {
  const headers = {
    'Content-Type': 'application/json',
    ...options.headers
  };

  if ((options.method === 'POST' || options.method === 'PUT') && options.data) {
    options.body = JSON.stringify(options.data);
    delete options.data;
  }
  
  return next({
    ...options,
    headers,
    credentials: 'include'
  });
});

if (window.pagespscData && window.pagespscData.restUrl) {
  apiFetch.use(apiFetch.createRootURLMiddleware(window.pagespscData.restUrl));
}

const StatusToggle = ({ isActive, onChange, disabled }) => {
  return createElement('button', {
    className: `cpl-status-toggle ${isActive ? 'is-active' : ''}`,
    onClick: onChange,
    disabled: disabled,
    type: 'button',
    'aria-label': isActive ? 'Deactivate' : 'Activate'
  },
    createElement('span', { className: 'cpl-status-toggle-track' }),
    createElement('span', { className: 'cpl-status-toggle-thumb' }),
    createElement('span', { className: 'cpl-status-indicator' })
  );
};

const AdminApp = () => {
  const [conditions, setConditions] = useState([]);
  const [plugins, setPlugins] = useState([]);
  const [pages, setPages] = useState([]);
  const [categories, setCategories] = useState([]);
  const [tags, setTags] = useState([]);
  const [loading, setLoading] = useState(true);
  const [saving, setSaving] = useState(false);
  const [showModal, setShowModal] = useState(false);
  const [editingCondition, setEditingCondition] = useState(null);
  const [notification, setNotification] = useState(null);
  const [searchTerm, setSearchTerm] = useState('');
  const [contentSearch, setContentSearch] = useState({ pages: '', categories: '', tags: '' });

  // Check for Pro data override
  const proData = window.cplProData || {};
  const isPremiumFromPro = proData.isPremium === true || proData.isPremium === '1' || proData.isPremium === 1;
  
  // Set freemium status with Pro override
  let freemiumStatus = window.pagespscData?.freemiumStatus || {
    is_premium: false,
    limits: { max_conditions: 2, pages_only: true },
    remaining: 2
  };
  
  // Override freemium status if Pro is active
  if (isPremiumFromPro) {
    freemiumStatus = {
      is_premium: true,
      limits: { max_conditions: null, pages_only: false },
      remaining: null
    };
  }
  

  useEffect(() => {
    loadData();
  }, []);

  const loadData = async () => {
    try {
      setLoading(true);
      
      const [conditionsData, pluginsData, contentData, statusData] = await Promise.all([
        apiFetch({ path: 'pagespsc/v1/conditions', method: 'GET' }),
        apiFetch({ path: 'pagespsc/v1/plugins', method: 'GET' }),
        apiFetch({ path: 'pagespsc/v1/content', method: 'GET' }),
        apiFetch({ path: 'pagespsc/v1/status', method: 'GET' })
      ]);
      
      setConditions(conditionsData || []);
      setPlugins(pluginsData || []);
      
      // Set individual content arrays
      const content = contentData || { pages: [], categories: [], tags: [] };
      setPages(content.pages || []);
      setCategories(content.categories || []);
      setTags(content.tags || []);
    } catch (error) {
      console.error('Load data error:', error);
      showNotification('Failed to load data. Please check your permissions.', 'error');
    } finally {
      setLoading(false);
    }
  };

  const showNotification = (message, type = 'success', opts = {}) => {
    setNotification({ message, type, ...opts });
    if (!opts.persistent) {
      setTimeout(() => setNotification(null), 3000);
    }
  };

  const handleCreateCondition = () => {
    // Check if user can create more conditions (Lite version limit: 2, Pro unlimited)
    if (!freemiumStatus.is_premium && conditions.length >= 2) {
      showNotification('Maximum conditions reached for Lite version.', 'info');
      return;
    }

    setEditingCondition({
      condition_name: '',
      plugins: [],
      include_exclude: 'exclude',
      pages: [],
      categories: [],
      tags: [],
      status: 'active'
    });
    setShowModal(true);
  };

  const handleEditCondition = (condition) => {
    setEditingCondition(condition);
    setShowModal(true);
  };

  const handleSaveCondition = async (conditionData) => {
    setSaving(true);
    try {
      // Lite version: Limit to 3 pages maximum for meaningful limitation
      if (!freemiumStatus.is_premium && conditionData.pages.length > 3) {
        showNotification('Maximum pages per condition reached for Lite version.', 'info');
        setSaving(false);
        return;
      }

      const cleanData = {
        condition_name: conditionData.condition_name || '',
        plugins: Array.isArray(conditionData.plugins) ? conditionData.plugins : [],
        include_exclude: conditionData.include_exclude || 'exclude', // Default to exclude for Lite
        pages: Array.isArray(conditionData.pages) ? conditionData.pages : [],
        pages_meta: Array.isArray(conditionData.pages_meta) ? conditionData.pages_meta : [],
        categories: freemiumStatus.is_premium ? (Array.isArray(conditionData.categories) ? conditionData.categories : []) : [], // Pro only
        tags: freemiumStatus.is_premium ? (Array.isArray(conditionData.tags) ? conditionData.tags : []) : [], // Pro only
        status: conditionData.status || 'active'
      };
      
      let response;
      if (editingCondition && editingCondition.id) {
        const formData = new FormData();
        formData.append('action', 'pagespsc_update_condition');
        formData.append('nonce', window.pagespscData.ajaxNonce);
        formData.append('id', editingCondition.id);
        formData.append('data', JSON.stringify(cleanData));
        
        const ajaxResponse = await fetch(window.pagespscData.ajaxUrl, {
          method: 'POST',
          body: formData,
          credentials: 'same-origin'
        });
        
        const result = await ajaxResponse.json();
        
        if (!result.success) {
          throw new Error(result.data || 'Update failed');
        }
        
        response = result.data;
      } else {
        const formData = new FormData();
        formData.append('action', 'pagespsc_create_condition');
        formData.append('nonce', window.pagespscData.ajaxNonce);
        formData.append('data', JSON.stringify(cleanData));
        
        const ajaxResponse = await fetch(window.pagespscData.ajaxUrl, {
          method: 'POST',
          body: formData,
          credentials: 'same-origin'
        });
        
        const result = await ajaxResponse.json();
        
        if (!result.success) {
          throw new Error(result.data || 'Create failed');
        }
        
        response = result.data;
      }
      
      setShowModal(false);
      loadData();
    } catch (error) {
      console.error('Save error:', error);
      const code = error && error.code || (error && error.data && error.data.code);
      const msg = (error && error.message) || (error && error.data && error.data.message) || error.message || 'Failed to save condition.';
      const upgradeUrl = 'https://wpcraftbench.com/product/plugin-load-manager/';
      if (code === 'limit_reached' || code === 'premium_feature' || (error && error.data && error.data.status === 403)) {
        showNotification(
          msg || 'Consider upgrading to Pro for additional features and unlimited conditions.',
          'info'
        );
      } else {
        showNotification(msg || 'Failed to save condition. Please check your permissions.', 'error');
      }
    } finally {
      setSaving(false);
    }
  };

  const handleDeleteCondition = async (id) => {
    if (!confirm('Are you sure you want to delete this condition?')) {
      return;
    }
    
    try {
      const formData = new FormData();
      formData.append('action', 'pagespsc_delete_condition');
      formData.append('nonce', window.pagespscData.ajaxNonce);
      formData.append('id', id);
      
      const ajaxResponse = await fetch(window.pagespscData.ajaxUrl, {
        method: 'POST',
        body: formData,
        credentials: 'same-origin'
      });
      
      const result = await ajaxResponse.json();
      
      if (!result.success) {
        throw new Error(result.data || 'Delete failed');
      }
      
      loadData();
      showNotification('Condition deleted successfully.', 'success');
    } catch (error) {
      console.error('Delete error:', error);
      showNotification('Failed to delete condition.', 'error');
    }
  };

  const handleToggleStatus = async (condition) => {
    try {
      const newStatus = condition.status === 'active' ? 'inactive' : 'active';
      
      try {
        await apiFetch({
          path: `pagespsc/v1/conditions/${condition.id}`,
          method: 'PUT',
          data: { status: newStatus }
        });
      } catch (putError) {
        await apiFetch({
          path: `pagespsc/v1/conditions/${condition.id}`,
          method: 'POST',
          data: { status: newStatus }
        });
      }

      loadData();
      showNotification(`Condition ${condition.status === 'active' ? 'deactivated' : 'activated'}`);
    } catch (error) {
      console.error('Toggle error:', error);
      const msg = (error && error.message) || (error && error.data && error.data.message) || 'Failed to update status';
      showNotification(`Failed to update status: ${msg}`, 'error');
    }
  };

  const filteredPlugins = plugins.filter(plugin => 
    plugin.name.toLowerCase().includes(searchTerm.toLowerCase())
  );

  const content = { pages, categories, tags };
  
  const filteredContent = {
    pages: content.pages.filter(page => 
      page.title.toLowerCase().includes(contentSearch.pages.toLowerCase())
    ),
    categories: content.categories.filter(cat => 
      cat.name.toLowerCase().includes(contentSearch.categories.toLowerCase())
    ),
    tags: content.tags.filter(tag => 
      tag.name.toLowerCase().includes(contentSearch.tags.toLowerCase())
    )
  };

  if (loading) {
    return createElement('div', { className: 'cpl-loading-container' },
      createElement(Spinner)
    );
  }

  const openUpgrade = (url) => {
    const upgradeUrl = 'https://wpcraftbench.com/product/plugin-load-manager/';
    window.open(upgradeUrl, '_blank');
  };

  return createElement('div', { className: 'cpl-admin-wrapper' },
    notification && createElement('div', { className: 'cpl-notification-container' },
      createElement(Notice, { 
        status: notification.type, 
        isDismissible: true,
        onRemove: () => setNotification(null),
        className: 'cpl-notification'
      }, 
        createElement(Fragment, null,
          createElement('span', null, notification.message),
          notification.upgradeUrl && createElement(Button, {
            variant: 'primary',
            className: 'cpl-upgrade-button is-green',
            onClick: () => openUpgrade(notification.upgradeUrl)
          }, 'Get Pro Version')
        )
      )
    ),

    createElement('div', { className: 'cpl-header' },
      createElement('div', { className: 'cpl-header-content' },
        createElement('div', { className: 'cpl-header-left' },
          createElement('h1', { className: 'cpl-main-title' }, 'Page Speed Script Optimizer'),
          createElement('p', { className: 'cpl-subtitle' }, 
            'Lite version — create up to 2 conditions. Pages targeting included. Get Pro for unlimited conditions and advanced targeting.'
          )
        ),
        createElement('div', { className: 'cpl-header-right' },
          createElement(Button, { 
            variant: 'secondary',
            className: 'cpl-create-button',
            onClick: handleCreateCondition 
          },
            createElement('span', { className: 'dashicons dashicons-plus-alt2' }),
            'Create Condition'
          )
        )
      )
    ),

    createElement('div', { className: 'cpl-content' },
      conditions.length === 0 ? 
        createElement('div', { className: 'cpl-empty-state' },
          createElement('div', { className: 'cpl-empty-state-icon' },
            createElement('span', { className: 'dashicons dashicons-admin-plugins' })
          ),
          createElement('h2', {}, 'No conditions yet'),
          createElement('p', {}, 'Create your first condition to start optimizing plugin loading'),
          createElement(Button, { 
            variant: 'secondary',
            className: 'cpl-create-button',
            onClick: handleCreateCondition 
          },
            createElement('span', { className: 'dashicons dashicons-plus-alt2' }),
            'Create Your First Condition'
          )
        ) : 
        createElement('div', { className: 'cpl-conditions-grid' },
          conditions.map((condition) => 
            createElement('div', { 
              key: condition.id,
              className: `cpl-condition-card ${condition.status === 'active' ? 'is-active' : 'is-inactive'}`
            },
              createElement('div', { className: 'cpl-condition-header' },
                createElement('div', { className: 'cpl-condition-title-row' },
                  createElement('h3', { className: 'cpl-condition-title' }, condition.condition_name),
                  createElement(StatusToggle, {
                    isActive: condition.status === 'active',
                    onChange: () => handleToggleStatus(condition)
                  })
                ),
                createElement('div', { className: 'cpl-condition-meta' },
                  createElement('span', { 
                    className: `cpl-condition-type ${condition.include_exclude}`
                  }, 
                    condition.include_exclude === 'include' ? 'Include' : 'Exclude'
                  ),
                  createElement('span', { className: 'cpl-plugin-count' },
                    `${condition.plugins.length} plugin${condition.plugins.length !== 1 ? 's' : ''}`
                  )
                )
              ),
              
              createElement('div', { className: 'cpl-condition-body' },
                createElement('div', { className: 'cpl-condition-section' },
                  createElement('h4', { className: 'cpl-section-title' }, 'Affected Plugins'),
                  createElement('div', { className: 'cpl-plugin-list' },
                    condition.plugins.map((pluginFile) => {
                      const plugin = plugins.find(p => p.file === pluginFile);
                      return createElement('div', { 
                        key: pluginFile,
                        className: 'cpl-plugin-item'
                      },
                        createElement('span', { className: 'cpl-plugin-icon dashicons dashicons-admin-plugins' }),
                        createElement('span', { className: 'cpl-plugin-name' },
                          plugin ? plugin.name : pluginFile
                        )
                      );
                    })
                  )
                ),
                
                ((condition.pages && condition.pages.length > 0) || (condition.categories && condition.categories.length > 0) || (condition.tags && condition.tags.length > 0)) &&
                createElement('div', { className: 'cpl-condition-section' },
                  createElement('h4', { className: 'cpl-section-title' }, 'Applied To'),
                  createElement('div', { className: 'cpl-applied-to' },
                    (condition.pages && condition.pages.length > 0) && 
                    createElement('div', { className: 'cpl-applied-item' },
                      createElement('span', { className: 'cpl-applied-icon dashicons dashicons-admin-page' }),
                      createElement('span', {}, 
                        `${condition.pages.length} page${condition.pages.length !== 1 ? 's' : ''}`
                      )
                    ),
                    (condition.categories && condition.categories.length > 0) && 
                    createElement('div', { className: 'cpl-applied-item' },
                      createElement('span', { className: 'cpl-applied-icon dashicons dashicons-category' }),
                      createElement('span', {}, 
                        `${condition.categories.length} categor${condition.categories.length !== 1 ? 'ies' : 'y'}`
                      )
                    ),
                    (condition.tags && condition.tags.length > 0) && 
                    createElement('div', { className: 'cpl-applied-item' },
                      createElement('span', { className: 'cpl-applied-icon dashicons dashicons-tag' }),
                      createElement('span', {}, 
                        `${condition.tags.length} tag${condition.tags.length !== 1 ? 's' : ''}`
                      )
                    )
                  )
                )
              ),
              
              createElement('div', { className: 'cpl-condition-footer' },
                createElement(Button, { 
                  variant: 'secondary',
                  className: 'cpl-edit-button',
                  onClick: () => handleEditCondition(condition) 
                },
                  createElement('span', { className: 'dashicons dashicons-edit' }),
                  'Edit'
                ),
                createElement(Button, { 
                  variant: 'link',
                  className: 'cpl-delete-button',
                  isDestructive: true,
                  onClick: () => handleDeleteCondition(condition.id)
                }, 'Delete')
              )
            )
          )
        )
    ),

    createElement('div', { className: 'cpl-content' },
      !freemiumStatus.is_premium && createElement('div', { className: 'cpl-upgrade-banner' },
        createElement('div', { className: 'cpl-upgrade-header' },
          createElement('div', { className: 'cpl-upgrade-icon' },
            createElement('span', { className: 'dashicons dashicons-star-filled' })
          ),
          createElement('div', { className: 'cpl-upgrade-title-section' },
            createElement('h3', { className: 'cpl-upgrade-title' }, 'Upgrade to Pro'),
            createElement('p', { className: 'cpl-upgrade-subtitle' },
              'Unlock unlimited conditions, advanced targeting, and premium features'
            )
          )
        ),
        createElement('div', { className: 'cpl-upgrade-content' },
          createElement('div', { className: 'cpl-compare' },
            createElement('div', { className: 'cpl-compare-col' },
              createElement('div', { className: 'cpl-compare-header' },
                createElement('span', { className: 'cpl-badge free' },
                  createElement('span', { className: 'dashicons dashicons-unlock' }), ' Lite'
                ),
                createElement('span', { className: 'cpl-text-muted' }, 'Current Version')
              ),
              createElement('ul', { className: 'cpl-compare-list' },
                createElement('li', { className: 'cpl-compare-item is-free' },
                  createElement('span', { className: 'dashicons dashicons-yes' }),
                  createElement('span', null, `Up to ${ (freemiumStatus && freemiumStatus.limits && freemiumStatus.limits.max_conditions) ? freemiumStatus.limits.max_conditions : 2 } conditions`)
                ),
                createElement('li', { className: 'cpl-compare-item is-free' },
                  createElement('span', { className: 'dashicons dashicons-admin-page' }),
                  createElement('span', null, 'Pages targeting')
                ),
                createElement('li', { className: 'cpl-compare-item is-missing' },
                  createElement('span', { className: 'dashicons dashicons-minus' }),
                  createElement('span', null, 'No Categories/Tags targeting')
                ),
                createElement('li', { className: 'cpl-compare-item is-missing' },
                  createElement('span', { className: 'dashicons dashicons-minus' }),
                  createElement('span', null, 'Limited performance optimization')
                )
              )
            ),
            createElement('div', { className: 'cpl-compare-col' },
              createElement('div', { className: 'cpl-compare-header' },
                createElement('span', { className: 'cpl-badge pro' },
                  createElement('span', { className: 'dashicons dashicons-lock' }), ' Pro'
                ),
                createElement('span', { className: 'cpl-text-muted' }, 'Everything in Free, plus')
              ),
              createElement('ul', { className: 'cpl-compare-list' },
                createElement('li', { className: 'cpl-compare-item is-pro' },
                  createElement('span', { className: 'dashicons dashicons-admin-multisite' }),
                  createElement('span', null, 'Unlimited conditions')
                ),
                createElement('li', { className: 'cpl-compare-item is-pro' },
                  createElement('span', { className: 'dashicons dashicons-category' }),
                  createElement('span', null, 'Categories & Tags targeting')
                ),
                createElement('li', { className: 'cpl-compare-item is-pro' },
                  createElement('span', { className: 'dashicons dashicons-performance' }),
                  createElement('span', null, 'Reduce server load, prevent heavy plugins where not needed')
                ),
                createElement('li', { className: 'cpl-compare-item is-pro' },
                  createElement('span', { className: 'dashicons dashicons-chart-bar' }),
                  createElement('span', null, 'Better Core Web Vitals/SEO and higher conversions')
                )
              )
            )
          ),
          createElement('p', { className: 'cpl-upgrade-footnote' }, 'Pro reduces plugin overhead on critical pages like homepage, landing pages, and checkout for better performance.')
        ),
        createElement('div', { className: 'cpl-upgrade-actions' },
          createElement(Button, {
            variant: 'primary',
            className: 'cpl-upgrade-button cpl-upgrade-button-prominent',
            onClick: () => openUpgrade('https://wpcraftbench.com/product/plugin-load-manager/')
          }, 'Upgrade to Pro')
        )
      )
    ),

    showModal && createElement(Modal, {
      title: (editingCondition && editingCondition.id) ? 'Edit Condition' : 'Create New Condition',
      onRequestClose: () => setShowModal(false),
      className: 'cpl-modal cpl-modal-wide'
    },
      createElement('div', { className: 'cpl-modal-container' },
        createElement('div', { className: 'cpl-modal-column cpl-modal-left' },
          createElement(VStack, { spacing: 4 },
            createElement(TextControl, {
              label: 'Condition Name',
              value: editingCondition.condition_name,
              onChange: (value) => setEditingCondition({ ...editingCondition, condition_name: value }),
              placeholder: 'e.g., Exclude heavy plugins from landing pages',
              className: 'cpl-input'
            }),

            createElement('div', { className: 'cpl-lite-limitation' },
              createElement('h4', { className: 'cpl-section-title' }, 'Action Type'),
              !freemiumStatus.is_premium && createElement('p', { className: 'cpl-lite-notice' }, 
                'Lite version: Exclude mode (prevents plugins from loading on selected pages).'
              ),
              createElement('div', { className: 'cpl-action-type-display' },
                createElement('span', { className: 'cpl-action-badge exclude' }, 'Exclude Mode')
              )
            ),

            createElement('div', { className: 'cpl-modal-section' },
              createElement('h3', { className: 'cpl-modal-section-title' }, 'Select Plugins'),
              createElement(SearchControl, {
                label: 'Search plugins',
                value: searchTerm,
                onChange: setSearchTerm,
                placeholder: 'Search by plugin name...'
              }),
              createElement('div', { className: 'cpl-plugin-selector cpl-scrollable-list' },
                filteredPlugins.map((plugin) => 
                  createElement('label', {
                    key: plugin.file,
                    className: 'cpl-plugin-checkbox'
                  },
                    createElement(CheckboxControl, {
                      checked: editingCondition.plugins.includes(plugin.file),
                      onChange: (checked) => {
                        const newPlugins = checked
                          ? [...editingCondition.plugins, plugin.file]
                          : editingCondition.plugins.filter(p => p !== plugin.file);
                        setEditingCondition({ ...editingCondition, plugins: newPlugins });
                      }
                    }),
                    createElement('div', { className: 'cpl-plugin-info' },
                      createElement('span', { className: 'cpl-plugin-name' }, plugin.name),
                      createElement('span', { className: 'cpl-plugin-status' }, 
                        plugin.active ? 'Active' : 'Inactive'
                      )
                    )
                  )
                )
              )
            )
          )
        ),

        createElement('div', { className: 'cpl-modal-column cpl-modal-right' },
          createElement('div', { className: 'cpl-modal-section' },
            createElement('h3', { className: 'cpl-modal-section-title' }, 'Target Content'),
            createElement(TabPanel, {
              className: 'cpl-content-tabs',
              activeClass: 'is-active',
              tabs: [
                {
                  name: 'pages',
                  title: 'Pages',
                  className: 'cpl-tab-pages'
                },
                {
                  name: 'categories',
                  title: 'Categories',
                  className: freemiumStatus.is_premium ? 'cpl-tab-categories' : 'cpl-tab-categories cpl-tab-disabled'
                },
                {
                  name: 'tags',
                  title: 'Tags',
                  className: freemiumStatus.is_premium ? 'cpl-tab-tags' : 'cpl-tab-tags cpl-tab-disabled'
                }
              ]
            }, (tab) => {
              if (tab.name === 'pages') {
                return createElement('div', { className: 'cpl-tab-content' },
                  !freemiumStatus.is_premium && createElement('p', { className: 'cpl-lite-notice' }, 
                    'Lite version: Pages targeting available.'
                  ),
                  createElement(SearchControl, {
                    label: 'Search pages',
                    value: contentSearch.pages,
                    onChange: (value) => setContentSearch({ ...contentSearch, pages: value }),
                    placeholder: 'Search pages...'
                  }),
                  createElement('div', { className: 'cpl-content-selector cpl-scrollable-list' },
                    // Select All Pages option (Pro only)
                    freemiumStatus.is_premium && createElement('label', {
                      className: 'cpl-content-checkbox cpl-select-all'
                    },
                      createElement(CheckboxControl, {
                        checked: editingCondition.pages.length === filteredContent.pages.length && filteredContent.pages.length > 0,
                        onChange: (checked) => {
                          const newPages = checked 
                            ? filteredContent.pages.map(page => page.id.toString())
                            : [];
                          setEditingCondition({ ...editingCondition, pages: newPages });
                        }
                      }),
                      createElement('span', { style: { fontWeight: 'bold' } }, 'Select All Pages')
                    ),
                    filteredContent.pages.map((page) => 
                      createElement('label', {
                        key: page.id,
                        className: 'cpl-content-checkbox'
                      },
                        createElement(CheckboxControl, {
                          checked: editingCondition.pages.includes(page.id.toString()),
                          onChange: (checked) => {
                            if (!freemiumStatus.is_premium && checked && editingCondition.pages.length >= 3) {
                              showNotification('Maximum pages per condition reached for Lite version.', 'info');
                              return;
                            }
                            const newPages = checked
                              ? [...editingCondition.pages, page.id.toString()]
                              : editingCondition.pages.filter(p => p !== page.id.toString());
                            setEditingCondition({ ...editingCondition, pages: newPages });
                          }
                        }),
                        createElement('span', {}, page.title)
                      )
                    )
                  )
                );
              }
              
              if (tab.name === 'categories') {
                if (!freemiumStatus.is_premium) {
                  return createElement('div', { className: 'cpl-tab-content cpl-disabled-content' },
                    createElement('div', { className: 'cpl-upgrade-prompt' },
                      createElement('div', { className: 'cpl-upgrade-icon' },
                        createElement('span', { className: 'dashicons dashicons-lock' })
                      ),
                      createElement('h4', {}, 'Categories Targeting - Pro Feature'),
                      createElement('p', {}, 'Target specific categories with your conditions. Upgrade to Pro to unlock this feature.'),
                      createElement(Button, {
                        variant: 'primary',
                        href: pagespscData.upgradeUrl,
                        target: '_blank'
                      }, 'Upgrade to Pro')
                    )
                  );
                }
                return createElement('div', { className: 'cpl-tab-content' },
                  createElement('p', { className: 'cpl-pro-notice' }, 
                    'Pro version: Target specific categories.'
                  ),
                  createElement(SearchControl, {
                    label: 'Search categories',
                    value: contentSearch.categories,
                    onChange: (value) => setContentSearch({ ...contentSearch, categories: value }),
                    placeholder: 'Search categories...'
                  }),
                  createElement('div', { className: 'cpl-content-selector cpl-scrollable-list' },
                    createElement('label', {
                      className: 'cpl-content-checkbox cpl-select-all'
                    },
                      createElement(CheckboxControl, {
                        checked: editingCondition.categories && editingCondition.categories.length === filteredContent.categories.length && filteredContent.categories.length > 0,
                        onChange: (checked) => {
                          const newCategories = checked 
                            ? filteredContent.categories.map(cat => cat.id.toString())
                            : [];
                          setEditingCondition({ ...editingCondition, categories: newCategories });
                        }
                      }),
                      createElement('span', { style: { fontWeight: 'bold' } }, 'Select All Categories')
                    ),
                    filteredContent.categories.map((category) => 
                      createElement('label', {
                        key: category.id,
                        className: 'cpl-content-checkbox'
                      },
                        createElement(CheckboxControl, {
                          checked: editingCondition.categories && editingCondition.categories.includes(category.id.toString()),
                          onChange: (checked) => {
                            const newCategories = checked
                              ? [...(editingCondition.categories || []), category.id.toString()]
                              : (editingCondition.categories || []).filter(c => c !== category.id.toString());
                            setEditingCondition({ ...editingCondition, categories: newCategories });
                          }
                        }),
                        createElement('span', {}, category.name)
                      )
                    )
                  )
                );
              }
              
              if (tab.name === 'tags') {
                if (!freemiumStatus.is_premium) {
                  return createElement('div', { className: 'cpl-tab-content cpl-disabled-content' },
                    createElement('div', { className: 'cpl-upgrade-prompt' },
                      createElement('div', { className: 'cpl-upgrade-icon' },
                        createElement('span', { className: 'dashicons dashicons-lock' })
                      ),
                      createElement('h4', {}, 'Tags Targeting - Pro Feature'),
                      createElement('p', {}, 'Target specific tags with your conditions. Upgrade to Pro to unlock this feature.'),
                      createElement(Button, {
                        variant: 'primary',
                        href: pagespscData.upgradeUrl,
                        target: '_blank'
                      }, 'Upgrade to Pro')
                    )
                  );
                }
                return createElement('div', { className: 'cpl-tab-content' },
                  createElement('p', { className: 'cpl-pro-notice' }, 
                    'Pro version: Target specific tags.'
                  ),
                  createElement(SearchControl, {
                    label: 'Search tags',
                    value: contentSearch.tags,
                    onChange: (value) => setContentSearch({ ...contentSearch, tags: value }),
                    placeholder: 'Search tags...'
                  }),
                  createElement('div', { className: 'cpl-content-selector cpl-scrollable-list' },
                    createElement('label', {
                      className: 'cpl-content-checkbox cpl-select-all'
                    },
                      createElement(CheckboxControl, {
                        checked: editingCondition.tags && editingCondition.tags.length === filteredContent.tags.length && filteredContent.tags.length > 0,
                        onChange: (checked) => {
                          const newTags = checked 
                            ? filteredContent.tags.map(tag => tag.id.toString())
                            : [];
                          setEditingCondition({ ...editingCondition, tags: newTags });
                        }
                      }),
                      createElement('span', { style: { fontWeight: 'bold' } }, 'Select All Tags')
                    ),
                    filteredContent.tags.map((tag) => 
                      createElement('label', {
                        key: tag.id,
                        className: 'cpl-content-checkbox'
                      },
                        createElement(CheckboxControl, {
                          checked: editingCondition.tags && editingCondition.tags.includes(tag.id.toString()),
                          onChange: (checked) => {
                            const newTags = checked
                              ? [...(editingCondition.tags || []), tag.id.toString()]
                              : (editingCondition.tags || []).filter(t => t !== tag.id.toString());
                            setEditingCondition({ ...editingCondition, tags: newTags });
                          }
                        }),
                        createElement('span', {}, tag.name)
                      )
                    )
                  )
                );
              }
            })
          )
        )
      ),

      createElement('div', { className: 'cpl-modal-footer' },
        createElement(Button, { 
          variant: 'tertiary',
          onClick: () => setShowModal(false) 
        }, 'Cancel'),
        createElement(Button, { 
          variant: 'primary',
          onClick: () => handleSaveCondition(editingCondition),
          isBusy: saving,
          disabled: !editingCondition.condition_name || editingCondition.plugins.length === 0
        }, (editingCondition && editingCondition.id) ? 'Update Condition' : 'Create Condition')
      )
    )
  );
};

document.addEventListener('DOMContentLoaded', () => {
  const root = document.getElementById('pagespsc-admin-root');
  if (root && window.wp && window.wp.element) {
    const { createRoot } = window.wp.element;
    createRoot(root).render(createElement(AdminApp));
  }
});
